/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.ant.tasks;

import java.io.Serializable;
import java.util.ArrayList;

import org.pepstock.catalog.Disposition;

/**
 * Is a container of dataset (one or more), addressable by name both a java code
 * (JNDI) and executable command (by environment variables). It could be defined
 * without datasets when is considered a sysout.<br>
 * There are 3 kinds of data description.<br>
 * <br>
 * <b>Sysout</b>: a file used in output way and in the same container of job-log
 * and message-log.<br>
 * Example:
 * <code>&lt;dataDescription name="FILEOUT" sysout="true" disposition="NEW"/&gt;</code>
 * <br>
 * <b>Single dataset</b>: a single dataset is defined which must accessible with
 * dispositions.<br>
 * Example: <br>
 * <code>&lt;dataDescription name="FILEIN" disposition="SHR"&gt;<br>
 *  		&lt;dataSet name="nas.rx.jemtest(0)"/&gt;<br>
 *  	 &lt;/dataDescription&gt;</code><br>
 * <b>Multi dataset</b>: more than one dataset are defined which must accessible
 * ONLY in SHR mode.<br>
 * Example: <code>&lt;dataDescription name="FILEIN" disposition="SHR"&gt;<br>
 *   		&lt;dataSet name="@@temp"/&gt;<br>
 *  		&lt;dataSet name="nas.rx.jemtest(0)"/&gt;<br>
 *       &lt;/dataDescription&gt;</code><br>
 * <br>
 * Defines also how to access to file.
 * 
 * @see org.pepstock.catalog.Disposition
 * @author Andrea "Stock" Stocchero
 * @version 1.0
 * 
 */
public class DataDescription implements Serializable {

	private static final long serialVersionUID = 1L;

	private ArrayList<DataSet> datasets = new ArrayList<DataSet>();

	private String name = null;

	private String disposition = Disposition.SHR;

	private boolean sysout = false;

	/**
	 * Empty constructor
	 */
	public DataDescription() {
	}

	/**
	 * Returns the name of data description. This is mandatory value because is
	 * used to access to resources by name.
	 * 
	 * @return the name of data description
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the name of data description. This is mandatory value because is
	 * used to access to resources by name.
	 * 
	 * @param name the name of data description
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Adds a dataset definition.
	 * 
	 * @param ds dataset definition
	 */
	public void addDataSet(DataSet ds) {
		datasets.add(ds);
	}

	/**
	 * Returns the list of datasets defined into data description
	 * 
	 * @return the list of datasets defined into data description
	 */
	public ArrayList<DataSet> getDatasets() {
		return datasets;
	}

	/**
	 * Returns the disposition (how to access to datasets)
	 * 
	 * @return the disposition (how to access to datasets)
	 */
	public String getDisposition() {
		return disposition;
	}

	/**
	 * Sets the disposition (how to access to datasets)
	 * 
	 * @param disposition the disposition (how to access to datasets)
	 */
	public void setDisposition(DispositionEnum disposition) {
		this.disposition = disposition.getValue();
	}

	/**
	 * Return <code>true</code> if the data description is defined with sysout
	 * attribute
	 * 
	 * @return <code>true</code> if the data description is defined with sysout
	 *         attribute
	 */
	public boolean isSysout() {
		return sysout;
	}

	/**
	 * Return <code>true</code> if the data description is defined with sysout
	 * attribute
	 * 
	 * @param sysout <code>true</code> if the data description is defined with
	 *            sysout attribute
	 */
	public void setSysout(boolean sysout) {
		this.sysout = sysout;
	}

	/**
	 * Return <code>true</code> if the size of datasets list is equals to 1.
	 * 
	 * @return <code>true</code> if the size of datasets list is equals to 1.
	 */
	public boolean isSingleDataset() {
		return (datasets.size() == 1);
	}

	/**
	 * Return <code>true</code> if the size of datasets list is greater to 1.
	 * 
	 * @return <code>true</code> if the size of datasets list is greater to 1.
	 */
	public boolean isMultiDataset() {
		return (datasets.size() > 1);
	}

	/**
	 * Returns the string representation of data description.
	 * 
	 * @return the string representation of data description
	 */
	@Override
	public String toString() {
		if (isSingleDataset()) {
			// if single, gets it and returns name, disposition and dataset
			// string
			DataSet ds = datasets.get(0);
			if (ds.isGdg()){
				return "[dd=" + getName() + ", disp=" + getDisposition() + ", " + ds.toString() + ", " + ds.getOffset() +"]";
			}
			return "[dd=" + getName() + ", disp=" + getDisposition() + ", " + ds.toString() + "]";
		} else if (!datasets.isEmpty()) {
			// if multi, gets it and returns name, disposition and all datasets
			// string
			StringBuffer sb = new StringBuffer("[dd=" + getName() + ", disp=" + getDisposition() + ",\n");
			for (DataSet ds : datasets) {
				sb.append("    ").append(ds.toString()).append(",").append('\n');
			}
			sb.append("]");
			return sb.toString();
		} else if (sysout) {
			// if sysout, gets it and returns name, disposition and "SYSOUT"
			// string
			return "[dd=" + getName() + ", disp=" + getDisposition() + ", SYSOUT]";
		}
		return super.toString();
	}

}