/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem;

import java.io.Serializable;

import com.google.gwt.user.client.rpc.GwtTransient;

/**
 * JCL (Job Control Language) is the class which contains all statements to
 * describe and execute jobs. It is possible to have different languages to
 * control jobs and for this reason is abstract about that.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class Jcl implements Serializable {

	private static final long serialVersionUID = 1L;
	
	/**
	 * Default affinity, value "default".
	 */
	public static final String DEFAULT_AFFINITY = "***";

	/**
	 * Default domain, value "default".
	 */
	public static final String DEFAULT_DOMAIN = "***";

	/**
	 * Default memory for the job, in MB
	 */
	public static final int DEFAULT_MEMORY = 128;

	/**
	 * Default priority for the job
	 */
	public static final int DEFAULT_PRIORITY = 10;
	
	/**
	 * Constant if JCL is not available. Happens often when you're looking at
	 * job in execution and it ends
	 */
	public static final String CONTENT_NOT_AVAILABLE = "JCL not available";

	private String type = null;

	/**
	 * Put @GWTTransinet to improve serialization performance. Do not use java transient because 
	 * JCL content must be serialize in Hazelcast but not in GWT
	 */
	@GwtTransient
	private String content = null;

	private String jobName = null;

	private String environment = null;

	private String domain = DEFAULT_DOMAIN;

	private String affinity = DEFAULT_AFFINITY;
	
	private int memory =  DEFAULT_MEMORY;

	private int priority = DEFAULT_PRIORITY;

	private boolean hold = false;
	
	/**
	 * Field containing the email addresses where to send the end
	 * Job notification. The addresses are semicolon separated.
	 */
	private String emailNotificationAddresses = null;
	
	private String user = null;

	/**
	 * Constructor without any arguments
	 */
	public Jcl() {
	}

	/**
	 * Sets the type of language (and then the factory) to parse and check the
	 * content of JCL by a short name defined on configuration.
	 * 
	 * @see org.pepstock.jem.factories#createJcl(String) factory creation method
	 * @param type type of job control language
	 */
	public void setType(String type) {
		this.type = type;
	}

	/**
	 * Returns the type of language of control jobs, or null if none.
	 * 
	 * @see org.pepstock.jem.factories#createJcl(String) factory creation method
	 * @return type of job control language
	 */
	public String getType() {
		return type;
	}

	/**
	 * Returns the name string for job, related to this JCL, or null if none.
	 * 
	 * @return the name string for job
	 */
	public String getJobName() {
		return jobName;
	}

	/**
	 * Sets the name string for job, related to this JCL after validation.
	 * 
	 * @param jobName the name string for job
	 */
	public void setJobName(String jobName) {
		this.jobName = jobName;
	}

	/**
	 * Sets the source code representing the JCL, by a string.
	 * 
	 * @param content the string representing source code
	 */
	public void setContent(String content) {
		this.content = content;
	}

	/**
	 * Returns the source code string, representing the JCL.
	 * 
	 * @return the string representing source code
	 */
	public String getContent() {
		return content;
	}

	/**
	 * Returns the name of cluster (see Hazelcast "group" definition) which the
	 * job must be executed in.
	 * 
	 * @return the name of cluster
	 */
	public String getEnvironment() {
		return environment;
	}

	/**
	 * Sets the name of cluster which the job must be executed in. It must be
	 * the same of Hazelcast group name otherwise the job will move on "routing"
	 * queue for another cluster.
	 * 
	 * @param environment the name of cluster
	 */
	public void setEnvironment(String environment) {
		this.environment = environment;
	}

	/**
	 * Returns the name of domain. Domain is a subset of nodes which job must be
	 * executed in.
	 * 
	 * @return the name of domain
	 */
	public String getDomain() {
		return domain;
	}

	/**
	 * Sets the name of domain which job must be executed in. If none of members
	 * is defined with this domain, job will stay in "input" queue, waiting for
	 * an initiator with this domain.
	 * 
	 * @param domain the name of domain
	 */
	public void setDomain(String domain) {
		this.domain = domain;
	}

	/**
	 * Sets the name of affinity which job must be executed in. If none of
	 * members is defined with this domain, job will stay in "input" queue,
	 * waiting for an initiator with this domain.
	 * 
	 * @return the name of affinity
	 */
	public String getAffinity() {
		return affinity;
	}

	/**
	 * Sets the name of affinity which job must be executed in. If none of
	 * members is defined with this domain, job will stay in "input" queue,
	 * waiting for an initiator with this domain.
	 * 
	 * @param affinity the name of affinity
	 */
	public void setAffinity(String affinity) {
		this.affinity = affinity;
	}
	
	/**
	 * Gets the email addresses where to send
	 * end Job notification email.
	 * 
	 * @return the email addresses for notification
	 */
	public String getEmailNotificationAddresses() {
		return emailNotificationAddresses;
	}

	/**
	 * Sets the email addresses where to send the
	 * end Job notification email.
	 * 
	 * @param emailNotificationAddresses the email addresses for notification, semicolon separated.
	 */
	public void setEmailNotificationAddresses(String emailNotificationAddresses) {
		this.emailNotificationAddresses = emailNotificationAddresses;
	}
	
	/**
	 * Return true if Jcl has the email addresses for notification: <br>
	 * checks if <code>emailNotificationAddresses</code> is different from null
	 * or an empty string.
	 *  
	 * @return true if Jcl has the email addresses for notification, false otherwise.
	 */
	public boolean hasEmailNotificationAddresses(){
		if(null == this.emailNotificationAddresses || 
			this.emailNotificationAddresses.trim().equalsIgnoreCase("")){
			return false;
		}
		return true;
	}
	
	/**
	 * Returns the memory estimated necessary to execute the job
	 * 
	 * @return the memory in MB
	 */
	public int getMemory() {
		return memory;
	}

	/**
	 * Sets the memory estimated necessary to execute the job
	 * 
	 * @param memory the memory in MB to set
	 */
	public void setMemory(int memory) {
		this.memory = memory;
	}

	/**
	 * Returns the priority for job, in "input" and "output" queues. Default is
	 * 10.
	 * 
	 * @return the priority of job when queued
	 */
	public int getPriority() {
		return priority;
	}

	/**
	 * Sets the priority for job, in "input" and "output" queues.
	 * 
	 * @param priority the priority of job when queued
	 */
	public void setPriority(int priority) {
		this.priority = priority;
	}

	/**
	 * Returns if job is blocked in "input" and "output" queues. Default is
	 * false.
	 * 
	 * @return true if job is hold, otherwise false
	 */
	public boolean isHold() {
		return hold;
	}

	/**
	 * Sets if job is blocked in "input" and "output" queues. Default is false.
	 * 
	 * @param hold true if job is hold, otherwise false
	 */
	public void setHold(boolean hold) {
		this.hold = hold;
	}

	/**
	 * Returns the user set inside of JCL.
	 * 
	 * @return the user
	 */
	public String getUser() {
		return user;
	}

	/**
	 * Sets the user to use during the submission of the job
	 * 
	 * @param user the user to set
	 */
	public void setUser(String user) {
		this.user = user;
	}
	
	
}