/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.catalog;

import java.io.Serializable;
import java.util.ArrayList;

/**
 * Is a container of datasets, addressable by name and with the disposition how
 * to access on datasets.
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0
 * 
 */
public class DataDescriptionImpl implements Serializable {

	private static final long serialVersionUID = 1L;

	private DataDescriptionImpl dataDescriptionReference = null;

	private String name = null;

	private String disposition = Disposition.SHR;

	private boolean sysout = false;

	private ArrayList<DataSetImpl> datasets = new ArrayList<DataSetImpl>();

	/**
	 * Empty constructor
	 */
	public DataDescriptionImpl() {

	}

	/**
	 * Returns the reference if exists, otherwise <code>null</code>. A reference
	 * occurs when the same data description is referenced in different steps of
	 * a job
	 * 
	 * @return referenced data description if exists, otherwise
	 *         <code>null</code>
	 */
	public DataDescriptionImpl getDataDescriptionReference() {
		return dataDescriptionReference;
	}

	/**
	 * Sets the reference for data description. A reference occurs when the same
	 * data description is referenced in different steps of a job
	 * 
	 * @param dataDescriptionReference reference data description instance
	 */
	public void setDataDescriptionReference(DataDescriptionImpl dataDescriptionReference) {
		this.dataDescriptionReference = dataDescriptionReference;
	}

	/**
	 * Returns the list of datasets.
	 * 
	 * @return the list of datasets
	 */
	public ArrayList<DataSetImpl> getDataSetsImpl() {
		return datasets;
	}

	/**
	 * Adds a dataset to the list.
	 * 
	 * @param dataSet dataset instance
	 */
	public void addDataSetImpl(DataSetImpl dataSet) {
		this.datasets.add(dataSet);
	}

	/**
	 * Returns the name of data description. This is mandatory value because is
	 * used to access to resources by name.
	 * 
	 * @return the name of data description
	 */
	public String getName() {
		return name;
	}

	/**
	 * Sets the name of data description. This is mandatory value because is
	 * used to access to resources by name.
	 * 
	 * @param name the name of data description
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * Returns the disposition (how to access to datasets)
	 * 
	 * @see Disposition
	 * @return the disposition (how to access to datasets)
	 */
	public String getDisposition() {
		return disposition;
	}

	/**
	 * Sets the disposition (how to access to datasets)
	 * 
	 * @see Disposition
	 * @param disposition the disposition (how to access to datasets)
	 */
	public void setDisposition(String disposition) {
		this.disposition = disposition;
	}

	/**
	 * Return <code>true</code> if the data description is defined with sysout
	 * attribute
	 * 
	 * @return <code>true</code> if the data description is defined with sysout
	 *         attribute
	 */
	public boolean isSysout() {
		return sysout;
	}

	/**
	 * Return <code>true</code> if the data description is defined with sysout
	 * attribute
	 * 
	 * @param sysout <code>true</code> if the data description is defined with
	 *            sysout attribute
	 */
	public void setSysout(boolean sysout) {
		this.sysout = sysout;
	}

	/**
	 * Return <code>true</code> if the size of datasets list is equals to 1.
	 * 
	 * @return <code>true</code> if the size of datasets list is equals to 1.
	 */
	public boolean isSingleDataset() {
		return (datasets.size() == 1);
	}

	/**
	 * Return <code>true</code> if the size of datasets list is greater to 1.
	 * 
	 * @return <code>true</code> if the size of datasets list is greater to 1.
	 */
	public boolean isMultiDataset() {
		return (datasets.size() > 1);
	}

	/**
	 * Returns the string representation of data description.
	 * 
	 * @return the string representation of data description
	 */
	@Override
	public String toString() {
		if (isSingleDataset()) {
			// if single, gets it and returns name, disposition and dataset
			// string
			DataSetImpl ds = datasets.get(0);
			return "[dd=" + getName() + ", disp=" + getDisposition() + ", " + ds.toString() + "]";
		} else if (!datasets.isEmpty()) {
			// if multi, gets it and returns name, disposition and all datasets
			// string
			StringBuffer sb = new StringBuffer("[dd=" + getName() + ", disp=" + getDisposition() + ",\n");
			for (DataSetImpl ds : datasets) {
				sb.append("    ").append(ds.toString()).append(",").append('\n');
			}
			sb.append("]");
			return sb.toString();
		} else if (sysout) {
			// if sysout, gets it and returns name, disposition and "SYSOUT"
			// string
			return "[dd=" + getName() + ", disp=" + getDisposition() + ", SYSOUT]";
		}
		// otherwise super.toString
		return super.toString();
	}

}