/*******************************************************************************
 * Copyright (c) 2012-2013 pepstock.org.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Enrico - initial API and implementation
 ******************************************************************************/
package org.pepstock.jem.plugin.views.tabbedQueues;

import java.util.ArrayList;
import java.util.Collection;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CTabFolder;
import org.eclipse.swt.custom.CTabItem;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.ISharedImages;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.pepstock.jem.Job;
import org.pepstock.jem.gwt.server.rest.entities.Jobs;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.plugin.Client;
import org.pepstock.jem.plugin.actions.LogOutAction;
import org.pepstock.jem.plugin.actions.SelectEnvAction;
import org.pepstock.jem.plugin.views.ColumnSortListener;
import org.pepstock.jem.plugin.views.JemViewBroker;
import org.pepstock.jem.plugin.views.JemViewPart;
import org.pepstock.jem.plugin.views.JobContentProvider;
import org.pepstock.jem.plugin.views.input.InputQueueTab;
import org.pepstock.jem.plugin.views.outline.OutlineView;
import org.pepstock.jem.plugin.views.output.OutputQueueTab;
import org.pepstock.jem.plugin.views.running.RunningQueueTab;


/**
 * @author Andrea "Stock" Stocchero
 *
 */
public class TabbedQueueView extends JemViewPart {

	/**
	 * The ID of the view as specified by the extension.
	 */
	public static final String ID = TabbedQueueView.class.getName();

	private ArrayList<QueueTab> tabs;
	private CTabFolder tabFolder;
	private Action doubleClickInspectAction;
	private Collection<Job> data = new ArrayList<Job>();
	
	@Override
	public void init(IViewSite site) throws PartInitException {
		super.init(site);
		JemViewBroker.getInstance().register(this);
	}
	
	/**
	 * The constructor.
	 */
	public TabbedQueueView() {
		tabs = new ArrayList<QueueTab>();
	}

	/**
	 * This is a callback that will allow us to create the viewer and initialize
	 * it.
	 */
	public void createPartControl(Composite parent) {
		tabFolder = new CTabFolder(parent, SWT.NONE);
		tabFolder.setToolTipText("Queues");
		tabs.add(new InputQueueTab(tabFolder, SWT.MULTI | SWT.V_SCROLL,this));
		tabs.add(new RunningQueueTab(tabFolder, SWT.MULTI | SWT.V_SCROLL,this));
		tabs.add(new OutputQueueTab(tabFolder, SWT.MULTI | SWT.V_SCROLL,this));
		tabFolder.setSelection(0);
		updateName();
		for (QueueTab tab : tabs) {
			createViewer(tab);
		}
		// Create the help context id for the viewer's control
		TableViewer[] tvs = getAllViewer();
		for (int i = 0; i < tvs.length; i++) {
			PlatformUI.getWorkbench().getHelpSystem().setHelp(tvs[i].getControl(), tvs[i].getClass().getName());
		}
		makeActions();
		hookContextMenu();
		hookDoubleClickAction();
		contributeToActionBars();
	}

	public void updateName(){
		String title = getTitle();
		title = title.indexOf('-')!=-1?title.substring(0,title.indexOf('-')):title;
		String name = Client.getInstance().getCurrentHost();
		name = name==null?title:title+"-"+name;
		setPartName(name);
	}
	
	public void addSearch(Composite parent, final QueueTab tab){
		Label searchLabel = new Label(parent, SWT.NONE);
		searchLabel.setText("Search: ");
		final Text searchText = new Text(parent, SWT.BORDER | SWT.SEARCH);
		searchText.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL | GridData.HORIZONTAL_ALIGN_FILL));
//		searchText.addKeyListener(new KeyListener() {
//			@Override
//			public void keyReleased(KeyEvent arg0) {
//				System.out.println("eee "+arg0.keyCode);
//				if(arg0.keyCode==13){
//					tab.refresh(searchText.getText());
//				}
//			}
//			
//			@Override
//			public void keyPressed(KeyEvent arg0) {
//			}
//		});
		searchText.addTraverseListener(new TraverseListener() {
			
			@Override
			public void keyTraversed(TraverseEvent arg0) {
				if(arg0.detail==SWT.TRAVERSE_RETURN){
					tab.refresh(searchText.getText());
				}				
			}
		});
		final Button reload = new Button(parent, SWT.SEARCH);
		reload.setImage(PlatformUI.getWorkbench().getSharedImages().getImage(ISharedImages.IMG_TOOL_REDO));
		reload.setText("Search");
		reload.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				tab.refresh(searchText.getText());
				super.widgetSelected(e);
			}
		});
	}
	
	/**
	 * @return
	 */
	public Collection<Job> getData(){
		return data;
	}

	/**
	 * @return
	 */
	public TableViewer getViewer() {
		CTabItem tis = tabFolder.getSelection();
		for (QueueTab tab : tabs) {
			if(tab.getName().equalsIgnoreCase(tis.getText())){
				return tab.getViewer();
			}
		}
		return null;
	}

	public TableViewer[] getAllViewer() {
		TableViewer[] tvs = new TableViewer[tabs.size()];
		for (int i=0; i<tabs.size();i++) {
			tvs[i]=tabs.get(i).getViewer();
		}
		return tvs;
	}

	private void createViewer(QueueTab tab) {
		tab.getViewer().setContentProvider(new JobContentProvider());
		tab.getViewer().setSorter(tab.getColumnSorter());
		//
		tab.createColumns(this);
		
		tab.getViewer().setLabelProvider(tab.getLabelProvider());
		tab.getViewer().setInput(data);
		
		final Table table = tab.getViewer().getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		// Make the selection available to other views
		getSite().setSelectionProvider(tab.getViewer());
		// Layout the viewer
		GridData gridData = new GridData();
		gridData.verticalAlignment = GridData.FILL;
		gridData.horizontalSpan = 3;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalAlignment = GridData.FILL;
		tab.getViewer().getControl().setLayoutData(gridData);
	}

	/**
	 * @param title
	 * @param bound
	 * @param colNumber
	 * @return
	 */
	public TableViewerColumn createTableViewerColumn(String title, int bound, final int colNumber, TableViewer viewer) {
		final TableViewerColumn viewerColumn = new TableViewerColumn(viewer, SWT.NONE);
		final TableColumn column = viewerColumn.getColumn();
		column.setText(title);
		column.setWidth(bound);
		column.setResizable(true);
		column.setMoveable(true);
		column.addSelectionListener(new ColumnSortListener(colNumber, viewer));
		return viewerColumn;
	}

	/**
	 * Passing the focus request to the viewer's control.
	 */
	public void setFocus() {
		getViewer().getControl().setFocus();
	}

	protected void hookContextMenu() {
		MenuManager menuMgr = new MenuManager("#PopupMenu");
		menuMgr.setRemoveAllWhenShown(true);
		menuMgr.addMenuListener(new IMenuListener() {
			public void menuAboutToShow(IMenuManager manager) {
				TabbedQueueView.this.fillContextMenu(manager);
			}
		});
		Menu menu = menuMgr.createContextMenu(getViewer().getControl());
		TableViewer[] tvs = getAllViewer();
		for (int i = 0; i < tvs.length; i++) {
			tvs[i].getControl().setMenu(menu);
			getSite().registerContextMenu(menuMgr, tvs[i]);
		}
	}

	private void fillContextMenu(IMenuManager manager) {
		manager.add(selectEnvAction);
		manager.add(logOutAction);
		// Other plug-ins can contribute there actions here
		manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
	}

	protected void makeActions() {
		selectEnvAction = new SelectEnvAction(this);
		logOutAction = new LogOutAction(this);
		doubleClickInspectAction = new Action() {
			public void run() {
				ISelection selection = getViewer().getSelection();
				Job job = (Job)((IStructuredSelection) selection).getFirstElement();
				if (job!=null){
					try {
						OutlineView	ow = (OutlineView)PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().showView(OutlineView.ID);
						Jobs jobs = new Jobs();
						jobs.setQueueName(getQueueName());
						ArrayList<Job> alj = new ArrayList<Job>();
						alj.add(job);
						jobs.setJobs(alj);
						ow.setData(Client.getInstance().getOutputTree(jobs),job);
						ow.setFocus();
					} catch (Exception e) {
						e.printStackTrace();
						MessageDialog.openError(getViewer().getControl().getShell(), "Error",e.getMessage());
					}
//					MessageDialog.openInformation(getViewer().getControl().getShell(), "JCL "+job.getName(),job.getJcl().getContent());
				}else
					MessageDialog.openWarning(getViewer().getControl().getShell(), "Warning","This job do not have any JCL");
			}
		};
	}

	private String getQueueName(){
		CTabItem tis = tabFolder.getSelection();
		if(Queues.OUTPUT_QUEUE.contains(tis.getText().toLowerCase()))
			return Queues.OUTPUT_QUEUE;
		if(Queues.RUNNING_QUEUE.contains(tis.getText().toLowerCase()))
			return Queues.RUNNING_QUEUE;
		return null;
	}
	
	private void hookDoubleClickAction() {
		TableViewer[] tvs = getAllViewer();
		for (int i = 0; i < tvs.length; i++) {
			tvs[i].addDoubleClickListener(new IDoubleClickListener() {
				public void doubleClick(DoubleClickEvent event) {
					doubleClickInspectAction.run();
				}
			});
		}
	}

	public CTabFolder getTabFolder() {
		return tabFolder;
	}

}
