/*******************************************************************************
 * Copyright (c) 2012-2013 pepstock.org.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Enrico - initial API and implementation
 ******************************************************************************/
package org.pepstock.jem.plugin.preferences;

import java.net.URI;
import java.util.HashMap;
import java.util.Iterator;

import org.apache.commons.lang3.StringUtils;
import org.eclipse.equinox.security.storage.StorageException;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.pepstock.jem.commands.util.HttpUtil;
import org.pepstock.jem.plugin.Activator;

public class TablePreferences extends PreferencePage implements
		IWorkbenchPreferencePage {

	
	private TableViewer tableViewer;
	private HashMap<String,Coordinate> tableModel;
	
	public TablePreferences() {
		super();
		this.noDefaultAndApplyButton();
		setPreferenceStore(Activator.getDefault().getPreferenceStore());
	}

	protected Control createContents(Composite parent) {
		Composite composite = new Composite(parent, SWT.NULL);
		composite.setLayout(new GridLayout(2, false));
		composite.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		Label listLabel = new Label(composite, SWT.NONE);
		listLabel.setText("&List of JEM environments:");
		new Label(composite, SWT.NONE);
		// Initializes TableViewer
		tableViewer = new TableViewer(composite, 
				SWT.V_SCROLL|SWT.H_SCROLL|SWT.BORDER|SWT.FULL_SELECTION);
		
		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setLayoutData(new GridData(GridData.FILL_BOTH));
		
		TableColumn column1 = new TableColumn(table, SWT.LEFT);
		column1.setText("Name");
		column1.setWidth(100);
		
		TableColumn column2 = new TableColumn(table, SWT.LEFT);
		column2.setText("Host");
		column2.setWidth(100);
		
		TableColumn column3 = new TableColumn(table, SWT.LEFT);
		column3.setText("Userid");
		column3.setWidth(100);

		TableColumn column4 = new TableColumn(table, SWT.LEFT);
		column4.setText("Password");
		column4.setWidth(100);

		TableColumn column5 = new TableColumn(table, SWT.LEFT);
		column5.setText("Default");
		column5.setWidth(100);

		tableViewer.setContentProvider(new IStructuredContentProvider() {
			
			@Override
			public void inputChanged(Viewer arg0, Object arg1, Object arg2) {
			}
			
			@SuppressWarnings("unchecked")
			@Override
			public Object[] getElements(Object arg0) {
				HashMap<String, Coordinate> input = (HashMap<String, Coordinate>) arg0;
				Coordinate[] res = new Coordinate[input.size()];
				int index = 0;
				for (Object current : input.values()) {
					res[index] = (Coordinate) current;
					index++;
				}
				return res;
			}

			@Override
			public void dispose() {
			}

		});
		tableViewer.setLabelProvider(new ITableLabelProvider(){
			public Image getColumnImage(Object element, int columnIndex) {
				return null;
			}
			public String getColumnText(Object element, int columnIndex) {
				Coordinate template = (Coordinate)element;
				switch(columnIndex){
				case 0: return template.getName();
				case 1: return template.getHost();
				case 2: return template.getUserid();
				case 3: return StringUtils.repeat("*", (int)(Math.random() * 16));
				case 4: return template.isDef()?"v":"";
				}
				return null;
			}
			public void addListener(ILabelProviderListener listener) {
			}
			public void dispose() {
			}
			public boolean isLabelProperty(Object element, String property) {
				return false;
			}
			public void removeListener(ILabelProviderListener listener) {
			}
		});
		tableViewer.setInput(tableModel);
		
		// Initializes Buttons
		Composite buttons = new Composite(composite, SWT.NULL);
		buttons.setLayout(new GridLayout(1, false));
		GridData gd = new GridData(GridData.VERTICAL_ALIGN_BEGINNING);
		gd.widthHint = 80;
		buttons.setLayoutData(gd);
		
		Button add = new Button(buttons, SWT.PUSH);
		add.setText("Add");
		add.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		add.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent evt){
				CoordinateDialog dialog = new CoordinateDialog(getShell());
				if(dialog.open()==Dialog.OK){
					tableModel.put(dialog.getCoordinate().getName(),dialog.getCoordinate());
					tableViewer.refresh();
				}
			}
		});
		
		final Button edit = new Button(buttons, SWT.PUSH);
		edit.setText("Edit");
		edit.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		edit.setEnabled(false);
		edit.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent evt){
				IStructuredSelection selection = (IStructuredSelection)tableViewer.getSelection();
				Coordinate template = (Coordinate)selection.getFirstElement();
				CoordinateDialog dialog = new CoordinateDialog(getShell(), template);
				if(dialog.open()==Dialog.OK){
					tableViewer.refresh();
				}
			}
		});
		
		final Button remove = new Button(buttons, SWT.PUSH);
		remove.setText("Remove");
		remove.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		remove.setEnabled(false);
		remove.addSelectionListener(new SelectionAdapter(){
			public void widgetSelected(SelectionEvent evt){
				IStructuredSelection selection = (IStructuredSelection)tableViewer.getSelection();
				tableModel.remove(((Coordinate)selection.getFirstElement()).getName());
				tableViewer.refresh();
			}
		});
		
		tableViewer.addSelectionChangedListener(new ISelectionChangedListener(){
			public void selectionChanged(SelectionChangedEvent event){
				if(tableViewer.getTable().getSelectionCount()==0){
					edit.setEnabled(false);
					remove.setEnabled(false);
				} else {
					edit.setEnabled(true);
					remove.setEnabled(true);
				}
			}
		});
		
		return composite;
	}

	private boolean hasDefault(){
		for (Iterator<String> iterator = tableModel.keySet().iterator(); iterator.hasNext();) {
			Coordinate type = tableModel.get(iterator.next());
			if(type.isDef())
				return true;
		}
		return false;
	}
	
	public void init(IWorkbench workbench) {
				try {
					tableModel = PreferenceManager.getInstance().loadPreferences();
				} catch (StorageException e) {
					MessageDialog.openError(getShell(),"Error","Error while loading preferences, your preferences may not be loaded: "+e.getMessage());
				} catch (Exception e) {
					MessageDialog.openError(getShell(),"Error","Error while initializing client, your preferences may not be loaded: "+e.getMessage());
				}
				
	}
	
	public boolean performOk() {
		try {
			PreferenceManager.getInstance().storePreferences(tableModel);
		} catch (StorageException e) {
			MessageDialog.openError(getShell(),"Error","Error while saving preferences, your preferences may not be saved: "+e.getMessage());
		} catch (Exception e) {
			MessageDialog.openError(getShell(),"Error","Error while initializing client, your preferences may not be saved: "+e.getMessage());
		}
		return super.performOk();
	}
	
	/**
	 * The dialog to create/edit templates.
	 */
	private class CoordinateDialog extends Dialog {

		private Text name;
		private Text host;
		private Text userid;
		private Text password;
		private Button def;
		private Coordinate coordinate;
		
		public CoordinateDialog(Shell parentShell) {
			this(parentShell, null);
		}

		public CoordinateDialog(Shell parentShell, Coordinate template) {
			super(parentShell);
			setShellStyle(getShellStyle()|SWT.RESIZE);
			this.coordinate = template;
		}
		
//		protected Point getInitialSize() {
//			return new Point(350,350);
//		}
		
		protected void createButtonsForButtonBar(Composite parent) {
			super.createButtonsForButtonBar(parent);
			updateButtonStatus();
		}
		
		protected Control createDialogArea(Composite parent) {
			getShell().setText("Coordinate");
			
			Composite main = new Composite(parent, SWT.LEFT);
			GridLayout mainLayout = new GridLayout(1, false);
			mainLayout.marginTop = 10;
			mainLayout.marginBottom = 10;
			mainLayout.marginLeft = 10;
			mainLayout.marginRight = 10;
			mainLayout.marginWidth = 0;
			mainLayout.marginHeight = 0;
			
			main.setLayout(mainLayout);
			main.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

			Label descLabel = new Label(main, SWT.LEFT);
			descLabel.setText("&Insert the JEM http host, userid and password:");

			Composite composite = new Composite(main, SWT.LEFT);
			GridLayout compLayout = new GridLayout(2, false);
			compLayout.marginTop = 10;
			compLayout.marginBottom = 10;
			compLayout.marginLeft = 0;
			compLayout.marginRight = 0;
			compLayout.marginWidth = 0;
			compLayout.marginHeight = 0;

			composite.setLayout(compLayout);
			composite.setLayoutData(new GridData(GridData.FILL_BOTH));

			Label nameLabel = new Label(composite, SWT.NONE);
			nameLabel.setText("&Name:");
			name = new Text(composite, SWT.BORDER);
			name.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			if(coordinate!=null){
				name.setText(coordinate.getName());
			}

			Label hostLabel = new Label(composite, SWT.NONE);
			hostLabel.setText("&Host:");
			host = new Text(composite, SWT.BORDER);
			host.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			if(coordinate!=null){
				host.setText(coordinate.getHost());
			}
			host.addModifyListener(new ModifyListener(){
				public void modifyText(ModifyEvent e){
					updateButtonStatus();
				}
			});

			Label useridLabel = new Label(composite, SWT.NONE);
			useridLabel.setText("&Userid:");
			userid = new Text(composite, SWT.BORDER);
			userid.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			if(coordinate!=null){
				userid.setText(coordinate.getUserid());
			}
			userid.addModifyListener(new ModifyListener(){
				public void modifyText(ModifyEvent e){
					updateButtonStatus();
				}
			});
			
			Label pwdLabel = new Label(composite, SWT.NONE);
			pwdLabel.setText("&Password:");
			password = new Text(composite, SWT.BORDER | SWT.PASSWORD);
			password.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
			if(coordinate!=null){
				password.setText(coordinate.getPassword());
			}
			password.addModifyListener(new ModifyListener(){
				public void modifyText(ModifyEvent e){
					updateButtonStatus();
				}
			});
			
			def = new Button(composite,SWT.CHECK);
			def.setText("&default");
			if(coordinate!=null){
				def.setSelection(coordinate.isDef());
			}
			else def.setSelection(!hasDefault());
			def.setToolTipText("Select default environment");
			return main;
		}
		
		private void updateButtonStatus(){
			boolean enabled = (host.getText().length()>0) && (userid.getText().length()>0) && (password.getText().length()>0); 
			getButton(IDialogConstants.OK_ID).setEnabled(enabled);
		}
		
		protected void okPressed() {
			String nome = name.getText();
			try {
				URI uri = URI.create(host.getText());
				if (uri.getScheme() != null){
					for (Iterator<String> i=tableModel.keySet().iterator(); i.hasNext();) {
						//overwrite the od one
//						if(name.equals(tableModel.get(i).getName())){
//							MessageDialog.openWarning(getParentShell(), "Warning", "Warning this environment name already exist, please enter a new one");
//							return;
//						}
						String n = i.next();
//						System.out.println(!n.equals(name.getText()));
						if(tableModel.get(n).isDef() && def.getSelection() && !n.equals(name.getText())){
							MessageDialog.openWarning(getParentShell(), "Warning", "Warning default environemnt already exist, please uncheck it before enter a new one");
							return;
						}
					} 
					nome = HttpUtil.getGroupName(host.getText());
				} else {
					throw new Exception("Invalid URI for host field: "+host.getText());
				}
            } catch (Exception e) {
            	String message = "Warning cannot connect to seleceted host cause: "+e.getMessage();
            	if(null == name.getText() || "".equals(name.getText())){
            		message = message+"\nIf the error persists, please enter a name to save preference.";
                	MessageDialog.openWarning(getShell(), "Warning", message);
    	            e.printStackTrace();
    				return;
            	}else{
            		message = message+"\nThis preference will be saved anyway with inserted name.";
                	MessageDialog.openWarning(getShell(), "Warning", message);
    	            e.printStackTrace();
            	}
            }
			if(coordinate==null){
				coordinate = new Coordinate();
			}
			coordinate.setName(nome);
			coordinate.setHost(host.getText());
			coordinate.setUserid(userid.getText());
			coordinate.setPassword(password.getText());
			coordinate.setDef(def.getSelection());
			super.okPressed();
			
		}
		
		public Coordinate getCoordinate(){
			return coordinate;
		}
	}

}
