/*******************************************************************************
 * Copyright (c) 2012-2013 pepstock.org.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Enrico - initial API and implementation
 ******************************************************************************/
package org.pepstock.jem.plugin;

import java.io.File;
import java.lang.management.ManagementFactory;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import javax.ws.rs.core.MediaType;

import org.pepstock.jem.GfsFile;
import org.pepstock.jem.Job;
import org.pepstock.jem.OutputFileContent;
import org.pepstock.jem.PreJob;
import org.pepstock.jem.commands.util.Factory;
import org.pepstock.jem.commands.util.HttpUtil;
import org.pepstock.jem.gwt.client.rest.GfsManager;
import org.pepstock.jem.gwt.client.rest.JobsManager;
import org.pepstock.jem.gwt.client.rest.LoginManager;
import org.pepstock.jem.gwt.client.security.LoggedUser;
import org.pepstock.jem.gwt.server.rest.entities.Account;
import org.pepstock.jem.gwt.server.rest.entities.GfsFileList;
import org.pepstock.jem.gwt.server.rest.entities.JobOutputListArgument;
import org.pepstock.jem.gwt.server.rest.entities.JobOutputTreeContent;
import org.pepstock.jem.gwt.server.rest.entities.Jobs;
import org.pepstock.jem.plugin.preferences.Coordinate;
import org.pepstock.jem.plugin.preferences.PreferenceManager;
import org.pepstock.jem.plugin.preferences.PreferenceNotFoundException;
import org.pepstock.jem.plugin.views.JemViewBroker;
import org.pepstock.jem.plugin.views.explorer.model.GfsFileModel;
import org.pepstock.jem.util.RestClient;

/**
 * @author Andrea "Stock" Stocchero
 *
 */
public class Client {
	
	private static Client INSTANCE = new Client();
	private Client changeEnv = null;
	public static final String OFFLINE = "favicon_bw.png";
	public static final String ONLINE = "favicon.png";
	
	private LoginManager man = new LoginManager();
	private JobsManager jobsMan = new JobsManager();
	private GfsManager gfsMan = new GfsManager();
	private LoggedUser user = null;
	private Coordinate current;
	private boolean logoff = false;
	
	/**
	 * @return
	 * @throws Exception
	 */
	public static Client getInstance(){
		return INSTANCE ;
	}
	
	private Client(){
	}

	private boolean isLoggedIn() throws PreferenceNotFoundException{
		if(!isLogged() && !logoff)
			login();
		return isLogged();
	}
	
	public boolean isLogged(){
		return user!=null;
	}

	private void login(Coordinate coordinate) throws PreferenceNotFoundException{
		try{
			RestClient.createInstance(coordinate.getHost()+"/rest");
			user = man.getUser();
			if (user == null) {
				// Creo l'oggetto account con uid e pwd
				Account account = new Account();
				account.setUserid(coordinate.getUserid());
				account.setPassword(coordinate.getPassword());
				// faccio login e mi salvo logged user
				user = man.login(account);
				current = coordinate;
				JemViewBroker.getInstance().setImageAll(Client.ONLINE);
			}
		} catch (Exception ex){
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw new PreferenceNotFoundException("Coordinate '"+coordinate.toString()+"' is default but there is this exception: "+ex.getMessage());
		}
	}
	
	private void login() throws PreferenceNotFoundException {
		try {
			HashMap<String, Coordinate> coordinates = PreferenceManager.getInstance().loadPreferences();
			if (coordinates.size() == 0)
				throw new PreferenceNotFoundException("JEM Environemnt preferences not found. Please insert preferences to start using JEM Plugin");
			for (Iterator<String> iterator = coordinates.keySet().iterator(); iterator.hasNext();) {
				Coordinate coordinate = coordinates.get(iterator.next());
				if (coordinate.isDef()) {
					try {
						login(coordinate);
						return;
					} catch (Exception e) {
						if (current != null)
							login(current);
						else
							throw new PreferenceNotFoundException("Coordinate '"+ coordinate.toString()+ "' is default but there is this exception: "+ e.getMessage());
					}
				}
			}
		} catch (Exception e) {
			throw new PreferenceNotFoundException("JEM Environemnt default preferences not found. Please insert preferences or select a default environment to start using JEM Plugin");
		}
		throw new PreferenceNotFoundException("JEM Environemnt default preferences not found. Please insert preferences or select a default environment to start using JEM Plugin");
	}

	public String getCurrentHost(){
		if(current!=null)
			return current.getName();
		return null;
	}
	
	public String loadCurrentHost() throws Exception{
		String uri = RestClient.getInstance().getBaseWebResource().getURI().toString();
		String host = uri.substring(0,uri.lastIndexOf('/'));
		String name;
		try {
			name = HttpUtil.getGroupName(host);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
		return name;
	}
	
	public boolean changeEnvironment(Coordinate coordinate) throws Exception{
		try{
			if(current!=null && isLogged()){
				String uri = RestClient.getInstance().getBaseWebResource().getURI().toString();
				String host = uri.substring(0,uri.lastIndexOf('/'));
				String name = HttpUtil.getGroupName(host);
				if(coordinate.getName().equalsIgnoreCase(name))
					return false;
				changeEnv = new Client();
				changeEnv.login(coordinate);
				logout();
				INSTANCE = changeEnv;
				changeEnv = null;
			}else
				login(coordinate);
			return true;
		}catch(Exception e){
			changeEnv = null;
			throw e;
		}
	}
	
	/**
	 * @param filter
	 * @return
	 * @throws Exception
	 */
	public Jobs refreshInput(String filter) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			Jobs jobs = jobsMan.getInputQueue(filter);
			return (jobs != null) ? jobs : new Jobs();
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	/**
	 * @param filter
	 * @return
	 * @throws Exception
	 */
	public Jobs refreshOutput(String filter) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			Jobs jobs = jobsMan.getOutputQueue(filter);
			return (jobs != null) ? jobs : new Jobs();
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	/**
	 * @param filter
	 * @return
	 * @throws Exception
	 */
	public Jobs refreshRunning(String filter) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			Jobs jobs = jobsMan.getRunningQueue(filter);
			return (jobs != null) ? jobs : new Jobs();
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	/**
	 * @param f
	 * @throws Exception
	 */
	public String submit(File jcl) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		URL url = jcl.toURI().toURL();
		PreJob preJob = Factory.createPreJob(url);
		preJob.setJclType("ant");// ant o sb
		Job job = new Job();
		job.setUser(user.getId());
		job.setInputArguments(ManagementFactory.getRuntimeMXBean().getInputArguments());
		preJob.setJob(job);
		RestClient.getInstance().getBaseWebResource().type(MediaType.APPLICATION_XML);
		try {
			return jobsMan.submit(preJob);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public JobOutputTreeContent getOutputTree(Jobs jobs) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			RestClient.getInstance().getBaseWebResource().type(MediaType.APPLICATION_XML);
			return jobsMan.getOutputTree(jobs);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public OutputFileContent getOutputFileContent(JobOutputListArgument jobOutputFileContet) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			return jobsMan.getOutputFileContent(jobOutputFileContet);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	public String getGfsFileData(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			return gfsMan.getFileData(path);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	public String getGfsFileLibrary(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			return gfsMan.getFileLibrary(path);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public String getGfsFileSource(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			return gfsMan.getFileSource(path);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public String getGfsFileClass(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			return gfsMan.getFileClass(path);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	public String getGfsFileBinary(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			return gfsMan.getFileBinary(path);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public String getGfsFile(GfsFileModel gfsfm) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		String sfile = "";
		switch (gfsfm.getType()) {
		case GfsFile.DATA:
			sfile = getGfsFileData(gfsfm.getLongName());
			break;
		case GfsFile.LIBRARY:
			sfile = getGfsFileLibrary(gfsfm.getLongName());
			break;
		case GfsFile.SOURCE:
			sfile = getGfsFileSource(gfsfm.getLongName());
			break;
		case GfsFile.CLASS:
			sfile = getGfsFileClass(gfsfm.getLongName());
			break;
		case GfsFile.BINARY:
			sfile = getGfsFileBinary(gfsfm.getLongName());
			break;

		default:
			break;
		}
		return sfile;
	}

	private List<GfsFileModel> getGfsModel(GfsFileList gfsfl, int type) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		Collection<GfsFile> collection = gfsfl.getGfsFiles();
		List<GfsFileModel> gfsL = new ArrayList<GfsFileModel>();
		if(collection!=null)
			for (Iterator<GfsFile> iterator = collection.iterator(); iterator.hasNext();) {
				GfsFile file = iterator.next();
				GfsFileModel gfsfm = new GfsFileModel(file, type);
				gfsL.add(gfsfm);
			}
		return gfsL;
	}
	
	public List<GfsFileModel> getGfsFileListData(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			GfsFileList gfsfl = gfsMan.getFilesListData(path);
			return getGfsModel(gfsfl,GfsFile.DATA);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	public List<GfsFileModel> getGfsFileListLibrary(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			GfsFileList gfsfl = gfsMan.getFilesListLibrary(path);
			return getGfsModel(gfsfl,GfsFile.LIBRARY);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public List<GfsFileModel> getGfsFileListSource(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			GfsFileList gfsfl = gfsMan.getFilesListSource(path);
			return getGfsModel(gfsfl,GfsFile.SOURCE);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}
	
	public List<GfsFileModel> getGfsFileListClass(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			GfsFileList gfsfl = gfsMan.getFilesListClass(path);
			return getGfsModel(gfsfl,GfsFile.CLASS);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

	public List<GfsFileModel> getGfsFileListBinary(String path) throws Exception{
		if(!isLoggedIn())
			throw new UserNotLoggedException("User not logged in. Please select an environment to login.");
		try{
			GfsFileList gfsfl = gfsMan.getFilesListBinary(path);
			return getGfsModel(gfsfl,GfsFile.BINARY);
		} catch (Exception e) {
			JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
			throw e;
		}
	}

/*	public List<GfsFileList> getGfsFileLists(String path) throws Exception{
		List<GfsFileList> all = new ArrayList<GfsFileList>();
		all.add(getGfsFileListData(path));
		all.add(getGfsFileListLibrary(path));
		all.add(getGfsFileListSource(path));
		all.add(getGfsFileListClass(path));
		all.add(getGfsFileListBinary(path));
		return all;
	}*/
	/**
	 * @throws Exception
	 */
	public void logout() throws Exception{
		if(man.getUser()!=null)
			man.logoff();
		user = null;
		logoff=true;
		JemViewBroker.getInstance().setImageAll(Client.OFFLINE);
	}
	
	
	
}
