/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package org.pepstock.jem.gwt.server.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.NodeInfoBean;
import org.pepstock.jem.gwt.server.UserInterfaceMessage;
import org.pepstock.jem.gwt.server.commons.SharedObjects;
import org.pepstock.jem.gwt.server.swarm.SwarmNodeMessage;
import org.pepstock.jem.gwt.server.swarm.SwarmQueues;
import org.pepstock.jem.gwt.server.swarm.executors.GetStatus;
import org.pepstock.jem.gwt.server.swarm.executors.Drain;
import org.pepstock.jem.gwt.server.swarm.executors.Start;
import org.pepstock.jem.node.NodeInfo;
import org.pepstock.jem.node.Status;
import org.pepstock.jem.node.configuration.SwarmConfiguration;
import org.pepstock.jem.node.security.Permissions;
import org.pepstock.jem.node.security.StringPermission;
import org.pepstock.jem.util.filters.Filter;
import org.pepstock.jem.util.filters.FilterToken;
import org.pepstock.jem.util.filters.fields.NodeFilterFields;
import org.pepstock.jem.util.filters.predicates.NodePredicate;

import com.hazelcast.core.Cluster;
import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.hazelcast.query.SqlPredicate;

/**
 * Is the manager of all operations to the nodes of JEM.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class SwarmNodesManager extends DefaultService {

	private NodesManager nodeManager = new NodesManager();

	/**
	 * Returns the list of all nodes joined the cluster.
	 * 
	 * @param nodesFilter
	 *            ipaddress or hostname filter
	 * @return collection of nodes
	 * @throws Exception
	 */
	public Collection<NodeInfoBean> getNodes(String nodesFilter) throws Exception {
		

		ArrayList<NodeInfoBean> list = new ArrayList<NodeInfoBean>();
		// if MainSwarm hazelcast instance is not running return empty list
		HazelcastInstance swarmHazelcast = SharedObjects.getInstance().getMainSwarm().getHazelcastInstance();
		if (swarmHazelcast == null || !swarmHazelcast.getLifecycleService().isRunning()) {
			return list;
		}
		// builds permission
		String permission = Permissions.SEARCH_NODES + nodesFilter.toLowerCase();
		// checks if the user is authorized to search nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(permission));

		// prepares SQL query to extract the right nodes
		String sqlFilter = nodesFilter.replace('.', '_');
		sqlFilter = sqlFilter.replace('*', '%');
		IMap<String, NodeInfo> nodes = swarmHazelcast.getMap(SwarmQueues.NODES_MAP);
		// creates SQL
		StringBuffer sb = new StringBuffer();
		sb.append("(hostname like '").append(sqlFilter).append("'").append(" OR ");
		sb.append("label like '").append(sqlFilter).append("') ").append(" AND ");
		sb.append("isSuperNode = false");
		SqlPredicate predicate = new SqlPredicate(sb.toString());

		// locks all map to have a consistent collection
		// only for 10 seconds otherwise 
		// throws an exception
		Collection<NodeInfo> allNodes = null;
		if (nodes.lockMap(10, TimeUnit.SECONDS)) {
			try {
				allNodes = nodes.values(predicate);
			} finally {
				// unlocks always teh map
				nodes.unlockMap();
			}
		} else {
			throw new Exception(UserInterfaceMessage.JEMG022E.toMessage().getFormattedMessage(SwarmQueues.NODES_MAP));
		}
		// must scan teh result of all nodes
		// becuse it has to serialize NodeInfoBean
		if (allNodes != null) {
			// gets the nodes info bean and returns them
			for (NodeInfo node : allNodes) {
				list.add(node.getNodeInfoBean());
			}
		}
		return list;
	}
	
	/**
	 * Returns the list of all nodes joined the cluster. UNKNOWN members are not returned
	 * 
	 * @param nodesFilter a String that will be parsed as a {@link Filter}
	 * @return collection of nodes
	 * @throws Exception  if any exception occurs
	 */
	public Collection<NodeInfoBean> getNodesByFilter(String nodesFilter) throws Exception {
		ArrayList<NodeInfoBean> list = new ArrayList<NodeInfoBean>();
		// if MainSwarm hazelcast instance is not running return empty list
		HazelcastInstance swarmHazelcast = SharedObjects.getInstance().getMainSwarm().getHazelcastInstance();
		if (swarmHazelcast == null || !swarmHazelcast.getLifecycleService().isRunning()) {
			return list;
		}
		// creates a filter object
		Filter filter = null;
		try {
			filter = Filter.parse(nodesFilter);
		} catch (Exception e) {
			// default case, all nodes
			filter = new Filter();
			filter.add(new FilterToken(NodeFilterFields.NAME.getName(), StringUtils.EMPTY));
		}
		// extract the label or hostname, if it is.
		// necessary to check permission because it is based on
		// label or hostname
		String nodesPermString = filter.get(NodeFilterFields.NAME.getName());
		// if label is null, try with hostname
		if ((nodesPermString == null) || (nodesPermString.trim().length() == 0)) {
			nodesPermString = filter.get(NodeFilterFields.HOSTNAME.getName());
			// if hostname is null as well, then use *
			if ((nodesPermString == null) || (nodesPermString.trim().length() == 0)) {
				nodesPermString = "*";
			}
		}
		// creates the right permission by jlabel or hostname
		String permission = Permissions.SEARCH_NODES+nodesPermString.toLowerCase();
		// checks if the user is authorized to get nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(permission));
		
		IMap<String, NodeInfo> nodes = swarmHazelcast.getMap(SwarmQueues.NODES_MAP);
		// creates predicate
		NodePredicate predicate = new NodePredicate(filter);

		Collection<NodeInfo> allNodes = null;
		// locks the whole map
		// if is not able to lock it in 10 seconds
		// throws an exception
		if (nodes.lockMap(10, TimeUnit.SECONDS)){ 
			try {
				// gets all nodes by predicate
				allNodes = nodes.values(predicate);
			} finally {
				// unlocks always the map
				nodes.unlockMap();
			}
		} else {
			throw new Exception(UserInterfaceMessage.JEMG022E.toMessage().getFormattedMessage(SwarmQueues.NODES_MAP));
		}
		if (allNodes != null){
			// gets the nodes and returns them
			// removing the nodes in UNKNOW
			// to avoid misunderstanding on UI
			for (NodeInfo node : allNodes){
				if (!node.getStatus().equals(Status.DRAINED))
					list.add(node.getNodeInfoBean());
			}
		}
		return list;
	}


	/**
	 * Starts swarm nodes, using a future task by executor service of Hazelcast.
	 * 
	 * @param nodes
	 *            nodes list of members to start
	 * @return always TRUE
	 * @throws Exception
	 */
	public Boolean start() throws Exception {
		// checks if the user is authorized to start nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.SWARM_NODES_START));
		
		SwarmConfiguration conf = SharedObjects.getInstance().getMainSwarm().getActiveConfiguration();
		if (!conf.isEnabled()){
			throw new Exception(SwarmNodeMessage.JEMO019E.toMessage().getFormattedMessage());
		}
		
		Boolean result = Boolean.TRUE;

		Collection<NodeInfoBean> webNodes = nodeManager.getWebNodes("*");
		// scans all nodes
		for (NodeInfoBean node : webNodes) {
			// gets the cluster to have member object of Hazelcast
			// to execute the future task
			Cluster cluster = getInstance().getCluster();
			// gets all members and scans them
			Set<Member> set = cluster.getMembers();
			for (Member member : set) {
				String memberKey = member.getUuid();
				// is the same member
				if (node.getKey().equalsIgnoreCase(memberKey)) {
					// creates the future task
					DistributedTask<Status> task = new DistributedTask<Status>(new Start(), member);
					// gets executor service and executes!
					ExecutorService executorService = getInstance().getExecutorService();
					executorService.execute(task);
					Status status;
					try {
						// gets result
						status = task.get();
						if (!status.equals(Status.ACTIVE)){
							result = Boolean.FALSE;
						}
					} catch (InterruptedException e) {
						throw new Exception(e.getMessage(), e);
					} catch (ExecutionException e) {
						throw new Exception(e.getMessage(), e);
					}
				}
			}
		}
		return result;
	}

	/**
	 * Shuts down all the swarm nodes, using a future task by executor service
	 * of Hazelcast.
	 * 
	 * @param nodes
	 *            list of nodes
	 * @return always true
	 * @throws Exception
	 */
	public Boolean drain() throws Exception {
		// checks if the user is authorized to stop nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.SWARM_NODES_DRAIN));
		Boolean result = Boolean.TRUE;

		Collection<NodeInfoBean> webNodes = nodeManager.getWebNodes("*");
		// scans all nodes
		for (NodeInfoBean node : webNodes) {
			Cluster cluster = getInstance().getCluster();
			// gets all members and scans them
			Set<Member> set = cluster.getMembers();
			for (Member member : set) {
				String memberKey = member.getUuid();
				// is the same member
				if (node.getKey().equalsIgnoreCase(memberKey)) {
					// creates the future task
					DistributedTask<Status> task = new DistributedTask<Status>(new Drain(), member);
					// gets executor service and executes!
					ExecutorService executorService = getInstance().getExecutorService();
					executorService.execute(task);
					Status status;
					try {
						// gets result
						status = task.get();
						if (!status.equals(Status.DRAINED)){
							result = Boolean.FALSE;
						}
					} catch (InterruptedException e) {
						throw new Exception(e.getMessage(), e);
					} catch (ExecutionException e) {
						throw new Exception(e.getMessage(), e);
					}
				}
			}
		}
		return result;
	}
	
	/**
	 * Returns the status of swarm 
	 * @return status if swarm
	 * @throws Exception if any exception occurs
	 */
	public String getStatus() throws Exception {
		// checks if the user is authenticated
		checkAuthentication();

		Status currentStatus = null;
		
		Collection<NodeInfoBean> webNodes = nodeManager.getWebNodes("*");
		// scans all nodes
		for (NodeInfoBean node : webNodes) {
			// gets the cluster to have member object of Hazelcast
			// to execute the future task
			Cluster cluster = getInstance().getCluster();
			// gets all members and scans them
			Set<Member> set = cluster.getMembers();
			for (Member member : set) {
				String memberKey = member.getUuid();
				// is the same member
				if (node.getKey().equalsIgnoreCase(memberKey)) {
					// creates the future task
					DistributedTask<Status> task = new DistributedTask<Status>(new GetStatus(), member);
					// gets executor service and executes!
					ExecutorService executorService = getInstance().getExecutorService();
					executorService.execute(task);
					Status status;
					try {
						// gets result
						status = task.get();
						if (currentStatus == null){
							currentStatus = status;
						} else {
							if (!currentStatus.equals(status)){
								return Status.UNKNOWN.getDescription();
							}
						}
					} catch (InterruptedException e) {
						throw new Exception(e.getMessage(), e);
					} catch (ExecutionException e) {
						throw new Exception(e.getMessage(), e);
					}
				}
			}
		}
		return (currentStatus == null) ? Status.UNKNOWN.getDescription() : currentStatus.getDescription();
	}

}