/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Marco "Cuc" Cuccato
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.NodeInfoBean;
import org.pepstock.jem.gwt.server.UserInterfaceMessage;
import org.pepstock.jem.node.ConfigurationFile;
import org.pepstock.jem.node.NodeInfo;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.Status;
import org.pepstock.jem.node.affinity.Result;
import org.pepstock.jem.node.configuration.ConfigKeys;
import org.pepstock.jem.node.executors.ExecutionResult;
import org.pepstock.jem.node.executors.GenericCallBack;
import org.pepstock.jem.node.executors.affinity.CheckAffinityPolicy;
import org.pepstock.jem.node.executors.affinity.GetAffinityPolicy;
import org.pepstock.jem.node.executors.affinity.SaveAffinityPolicy;
import org.pepstock.jem.node.executors.configuration.CheckHazelcastConfiguration;
import org.pepstock.jem.node.executors.configuration.CheckJemConfiguration;
import org.pepstock.jem.node.executors.configuration.CheckJemEnvConfiguration;
import org.pepstock.jem.node.executors.configuration.GetHazelcastConfiguration;
import org.pepstock.jem.node.executors.configuration.GetJemConfiguration;
import org.pepstock.jem.node.executors.configuration.GetJemEnvConfiguration;
import org.pepstock.jem.node.executors.configuration.SaveHazelcastConfiguration;
import org.pepstock.jem.node.executors.configuration.SaveJemConfiguration;
import org.pepstock.jem.node.executors.configuration.SaveJemEnvConfiguration;
import org.pepstock.jem.node.executors.nodes.Drain;
import org.pepstock.jem.node.executors.nodes.GetLog;
import org.pepstock.jem.node.executors.nodes.Shutdown;
import org.pepstock.jem.node.executors.nodes.Start;
import org.pepstock.jem.node.executors.nodes.Top;
import org.pepstock.jem.node.executors.nodes.Update;
import org.pepstock.jem.node.security.Permissions;
import org.pepstock.jem.node.security.StringPermission;
import org.pepstock.jem.util.filters.Filter;
import org.pepstock.jem.util.filters.FilterToken;
import org.pepstock.jem.util.filters.fields.NodeFilterFields;
import org.pepstock.jem.util.filters.predicates.NodePredicate;

import com.hazelcast.core.Cluster;
import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.hazelcast.query.SqlPredicate;


/**
 * Is the manager of all operations to the nodes of JEM. UNKNOWN members are not returned.
 * 
 * @author Marco "Cuc" Cuccato
 *
 */
public class NodesManager extends DefaultService {
	
	/**
	 * Returns the list of all normal nodes joined the cluster. UNKNOWN members are not returned.
	 * 
	 * @param nodesFilter filter contains all tokens to performs filtering
	 * @return collection of nodes
	 * @throws Exception if any exception occurs
	 */
	public Collection<NodeInfoBean> getNodes(String nodesFilter) throws Exception{
		return getNodes(nodesFilter, false);
	}

	/**
	 * Returns the list of all web nodes joined the cluster. UNKNOWN members are not returned
	 * 
	 * @param nodesFilter filter contains all tokens to performs filtering
	 * @return collection of nodes
	 * @throws Exception if any exception occurs 
	 */
	public Collection<NodeInfoBean> getWebNodes(String nodesFilter) throws Exception{
		return getNodes(nodesFilter, true);
	}

	/**
	 * Returns the list of all nodes joined the cluster. UNKNOWN members are not returned
	 * 
	 * @param nodesFilter  filter contains all tokens to performs filtering
	 * @param webNodes if true return only web nodes, if false return only nodes that are not web nodes
	 * @return collection of nodes
	 * @throws Exception if any exception occurs
	 */
	public Collection<NodeInfoBean> getNodes(String nodesFilter, boolean webNodes) throws Exception{
		// builds permission
		String permission = Permissions.SEARCH_NODES+nodesFilter.toLowerCase();
		// checks if the user is authorized to get nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(permission));
		
		// prepares SQL query to extract the right nodes
		String sqlFilter = nodesFilter.replace('.', '_');
		sqlFilter = sqlFilter.replace('*', '%');
		IMap<String, NodeInfo> nodes = getInstance().getMap(Queues.NODES_MAP);
		// creates SQL
		StringBuffer sb = new StringBuffer();
		sb.append("(hostname like '").append(sqlFilter).append("'").append(" OR ");
		sb.append("label like '").append(sqlFilter).append("') ").append(" AND ");
		sb.append("isSuperNode = " + webNodes);
		SqlPredicate predicate = new SqlPredicate(sb.toString());

		ArrayList<NodeInfoBean> list = new ArrayList<NodeInfoBean>();
		Collection<NodeInfo> allNodes = null;
		if (nodes.lockMap(10, TimeUnit.SECONDS)){ 
			try {
				allNodes = nodes.values(predicate);
			} finally {
				nodes.unlockMap();
			}
		} else {
			throw new Exception(UserInterfaceMessage.JEMG022E.toMessage().getFormattedMessage(Queues.NODES_MAP));
		}
		if (allNodes != null){
			// gets the nodes and returns them
			for (NodeInfo node : allNodes){
				if (!node.getStatus().equals(Status.UNKNOWN))
					list.add(node.getNodeInfoBean());
			}
		}
		// if list is not empty
		if (!list.isEmpty()){
			// sorts the list for KEY
			Collections.sort(list, new Comparator<NodeInfoBean>() {
				@Override
                public int compare(NodeInfoBean o1, NodeInfoBean o2) {
	                return o1.getKey().compareTo(o2.getKey());
                }
			});
		}
		return list;
	}
	
	/**
	 * Returns the list of all nodes joined the cluster. UNKNOWN members are not returned
	 * 
	 * @param nodesFilter a String that will be parsed as a {@link Filter}
	 * @return collection of nodes
	 * @throws Exception  if any exception occurs
	 */
	public Collection<NodeInfoBean> getNodesByFilter(String nodesFilter) throws Exception {
		// creates a filter object
		Filter filter = null;
		try {
			filter = Filter.parse(nodesFilter);
		} catch (Exception e) {
			// default case, all nodes
			filter = new Filter();
			filter.add(new FilterToken(NodeFilterFields.NAME.getName(), StringUtils.EMPTY));
		}
		// extract the label or hostname, if it is.
		// necessary to check permission because it is based on
		// label or hostname
		String nodesPermString = filter.get(NodeFilterFields.NAME.getName());
		// if label is null, try with hostname
		if ((nodesPermString == null) || (nodesPermString.trim().length() == 0)) {
			nodesPermString = filter.get(NodeFilterFields.HOSTNAME.getName());
			// if hostname is null as well, then use *
			if ((nodesPermString == null) || (nodesPermString.trim().length() == 0)) {
				nodesPermString = "*";
			}
		}
		// creates the right permission by jlabel or hostname
		String permission = Permissions.SEARCH_NODES+nodesPermString.toLowerCase();
		// checks if the user is authorized to get nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(permission));
		
		IMap<String, NodeInfo> nodes = getInstance().getMap(Queues.NODES_MAP);
		// creates predicate
		NodePredicate predicate = new NodePredicate(filter);

		ArrayList<NodeInfoBean> list = new ArrayList<NodeInfoBean>();
		Collection<NodeInfo> allNodes = null;
		// locks the whole map
		// if is not able to lock it in 10 seconds
		// throws an exception
		if (nodes.lockMap(10, TimeUnit.SECONDS)){ 
			try {
				// gets all nodes by predicate
				allNodes = nodes.values(predicate);
			} finally {
				// unlocks always the map
				nodes.unlockMap();
			}
		} else {
			throw new Exception(UserInterfaceMessage.JEMG022E.toMessage().getFormattedMessage(Queues.NODES_MAP));
		}
		if (allNodes != null){
			// gets the nodes and returns them
			// removing the nodes in UNKNOW
			// to avoid misunderstanding on UI
			for (NodeInfo node : allNodes){
				if (!node.getStatus().equals(Status.UNKNOWN))
					list.add(node.getNodeInfoBean());
			}
		}
		return list;
	}

	/**
	 * Drains the list of members, using a future task by executor service of Hazelcast. 
	 * 
	 * @param nodes list of members to drain 
	 * @return  always TRUE
	 * @throws Exception  if any exception occurs
	 */
	public Boolean drain(Collection<NodeInfoBean> nodes) throws Exception{
		// checks if the user is authorized to drain nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.NODES_DRAIN));
		// gets nodes map instance 
		IMap<String, NodeInfo> members_map = getInstance().getMap(Queues.NODES_MAP);

		Boolean result = Boolean.TRUE;

		// scans all nodes
		for (NodeInfoBean node : nodes){
			// is not a super node and is not UNKNOWN
			if (!node.isSuperNode() && !node.getStatus().equalsIgnoreCase(Status.UNKNOWN.getDescription())){
				// gets key
				String key = node.getKey();

				if (members_map.containsKey(key)){
					// gets the cluster to have member object of Hazelcast
					// to execute the future task
					Cluster cluster = getInstance().getCluster();
					// gets all members and scans them
					Set<Member> set = cluster.getMembers();
					for (Member member : set){
						String memberKey = member.getUuid();
						// is the same member
						if (node.getKey().equalsIgnoreCase(memberKey)){
							// creates the future task
							DistributedTask<ExecutionResult> task = new DistributedTask<ExecutionResult>(new Drain(), member);
							// gets executor service and executes!
							ExecutorService executorService = getInstance().getExecutorService();
							task.setExecutionCallback(new GenericCallBack());
							executorService.execute(task);
						} 
					}
				}
			}
		}
		return result;
	}
	
	/**
	 * Starts the list of members, using a future task by executor service of Hazelcast. 
	 * 
	 * @param nodes nodes list of members to start
	 * @return  always TRUE
	 * @throws Exception  if any exception occurs
	 */
	public Boolean start(Collection<NodeInfoBean> nodes) throws Exception{
		// checks if the user is authorized to start nodes
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.NODES_START));
		// gets nodes map instance 
		IMap<String, NodeInfo> members_map = getInstance().getMap(Queues.NODES_MAP);
		
		Boolean result = Boolean.TRUE;

		// scans all nodes
		for (NodeInfoBean node : nodes){
			// is not a super node and is not UNKNOWN
			if (!node.isSuperNode() && !node.getStatus().equalsIgnoreCase(Status.UNKNOWN.getDescription())){
				// gets key
				String key = node.getKey();
				// checks if is in map
				if (members_map.containsKey(key)){

					// gets the cluster to have member object of Hazelcast
					// to execute the future task
					Cluster cluster = getInstance().getCluster();
					// gets all members and scans them
					Set<Member> set = cluster.getMembers();
					for (Member member : set){
						String memberKey = member.getUuid();
						// is the same member
						if (node.getKey().equalsIgnoreCase(memberKey)){
							// creates the future task
							DistributedTask<ExecutionResult> task = new DistributedTask<ExecutionResult>(new Start(), member);
							// gets executor service and executes!
							ExecutorService executorService = getInstance().getExecutorService();
							task.setExecutionCallback(new GenericCallBack());
							executorService.execute(task);
						} 
					}

				}
			}
		}
		return result;
	}


	/**
	 * Shuts down the list of members, using a future task by executor service of Hazelcast.<br>
	 * Is not used.
	 *  
	 * @param nodes list of nodes
	 * @return always true
	 * @throws Exception  if any exception occurs
	 */
	@SuppressWarnings("unused")
    private Boolean shutdown(Collection<NodeInfoBean> nodes) throws Exception{
		checkAuthentication();
		// gets nodes map instance 
		IMap<String, NodeInfo> members_map = getInstance().getMap(Queues.NODES_MAP);

		Boolean result = Boolean.TRUE;

		// scans all nodes
		for (NodeInfoBean node : nodes){
			// is not a super node and is not UNKNOWN
			if (!node.isSuperNode() && !node.getStatus().equalsIgnoreCase(Status.UNKNOWN.getDescription())){
				// gets key
				String key = node.getKey();
				// checks if is in map
				if (members_map.containsKey(key)){

					// if is DRAINED or DRAINING than starts it 
					// gets the cluster to have member object of Hazelcast
					// to execute the future task
					Cluster cluster = getInstance().getCluster();
					// gets all members and scans them
					Set<Member> set = cluster.getMembers();
					for (Member member : set){
						String memberKey = member.getUuid();
						// is the same member
						if (node.getKey().equalsIgnoreCase(memberKey)){
							// creates the future task
							DistributedTask<ExecutionResult> task = new DistributedTask<ExecutionResult>(new Shutdown(), member);
							// gets executor service and executes!
							ExecutorService executorService = getInstance().getExecutorService();
							task.setExecutionCallback(new GenericCallBack());
							executorService.execute(task);
						} 
					}
				}
			}
		}
		return result;
	}

	/**
	 * Update the domain or static affinities of node
	 * @param node niode to update
	 * @return always true
	 * @throws Exception if any execption occurs
	 */
	public Boolean update(NodeInfoBean node) throws Exception{
		// checks if the user is authorized to update a node
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.NODES_UPDATE));
		
		Boolean result = Boolean.TRUE;
		
		Member member = getMember(node.getKey());
		// creates the future task
		DistributedTask<ExecutionResult> task = new DistributedTask<ExecutionResult>(new Update(node), member);		
		ExecutorService executorService = getInstance().getExecutorService();
		task.setExecutionCallback(new GenericCallBack());
		executorService.execute(task);
		
		return result;
	}
	
	/**
	 * Returns the configuration file for the node
	 * 
	 * @param node node where execute a future task to get the config file 
	 * @param what type of configuration file to return
	 * @return Configuration file container
	 * @throws Exception if exception occurs
	 */
	public ConfigurationFile getNodeConfigFile(NodeInfoBean node, String what) throws Exception{
		// checks if the user is authorized to read configuration
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_CONFIGURATION));
		
		ConfigurationFile result = null;
		
		Member member = getMember(node.getKey());
		
		// creates the future task
		DistributedTask<ConfigurationFile> task = null;

		// checks if wants Hazelcast or JEM node configuration
		// by default is JEM node configuration
		if (what != null){
			if (what.equalsIgnoreCase(ConfigKeys.JEM_CONFIG)){
				task = new DistributedTask<ConfigurationFile>(new GetJemConfiguration(), member);
			} else if (what.equalsIgnoreCase(ConfigKeys.AFFINITY)){
				task = new DistributedTask<ConfigurationFile>(new GetAffinityPolicy(), member);
			}
		} else {
			task = new DistributedTask<ConfigurationFile>(new GetJemConfiguration(), member);
		}

		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		try {
			// gets result
			result = task.get();
		} catch (Exception e) {
			throw new Exception(e.getMessage(), e);
		}
		return result;
	}

	/**
	 * Saves the configuration file for the node
	 * 
	 * @param node node where execute a future task to get the config file 
	 * @param file configuration file to save
	 * @param what type of configuration file to return
	 * @return Configuration file container
	 * @throws Exception if exception occurs
	 */
	public ConfigurationFile saveNodeConfigFile(NodeInfoBean node, ConfigurationFile file, String what) throws Exception{
		// checks if the user is authorized to read configuration
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_CONFIGURATION));
		
		ConfigurationFile result = null;
		
		Member member = getMember(node.getKey());
		
		// creates the future task
		DistributedTask<ConfigurationFile> task = null;

		// checks if wants Hazelcast or JEM node configuration
		// by default is JEM node configuration
		if (what != null){
			if (what.equalsIgnoreCase(ConfigKeys.JEM_CONFIG)){
				checkConfigFile(file.getContent(), what);
				task = new DistributedTask<ConfigurationFile>(new SaveJemConfiguration(file), member);
			} else if (what.equalsIgnoreCase(ConfigKeys.AFFINITY)){
				checkAffinityPolicy(node, file.getContent());
				task = new DistributedTask<ConfigurationFile>(new SaveAffinityPolicy(file), member);
			}
		} else {
			task = new DistributedTask<ConfigurationFile>(new SaveJemConfiguration(file), member);
		}

		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		try {
			// gets result
			result = task.get();
		} catch (Exception e) {
			throw new Exception(e.getMessage(), e);
		}
		return result;
	}

	/**
	 * Checks if syntax of content is correct.
	 * @param content content of configuration file
	 * @param what type of config file
	 * @return always true
	 * @throws Exception if any error parsing content occurs
	 */
	public Boolean checkConfigFile(String content, String what) throws Exception{
		// checks if the user is authorized to read configuration
		// if not, this method throws an exception
		try {
	        checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_CONFIGURATION));
        } catch (Exception e1) {
        	checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_CLUSTER_CONFIGURATION));
        }
		
		Member member = getMember();
		
		// creates the future task
		DistributedTask<Boolean> task = null;

		// checks if wants Hazelcast or JEM node configuration
		// by default is JEM node configuration
		if (what != null){
			if (what.equalsIgnoreCase(ConfigKeys.JEM_CONFIG)){
				task = new DistributedTask<Boolean>(new CheckJemConfiguration(content), member);
			} else if (what.equalsIgnoreCase(ConfigKeys.JEM_ENV_CONF)){
				task = new DistributedTask<Boolean>(new CheckJemEnvConfiguration(content), member);
			} else if (what.equalsIgnoreCase(ConfigKeys.HAZELCAST_CONFIG)){
				task = new DistributedTask<Boolean>(new CheckHazelcastConfiguration(content), member);
			}
		} else {
			task = new DistributedTask<Boolean>(new CheckJemConfiguration(content), member);
		}

		Boolean result = Boolean.TRUE;
		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		try {
			// gets result
			result = task.get();
		} catch (Exception e) {
			throw new Exception(e.getMessage(), e);
		}
		return result;
	}

	
	/**
	 * Checks if syntax of affinity loader policy content is correct.
	 * @param node node where execute a future task  
	 * @param content type of affinity policy
	 * @return always true
	 * @throws Exception if any error parsing content occurs
	 */
	public Result checkAffinityPolicy(NodeInfoBean node, String content) throws Exception{
		// checks if the user is authorized to read configuration
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_CONFIGURATION));
		
		Member member = getMember(node.getKey());
		
		Result result = null;
		
		// creates the future task
		DistributedTask<Result> task = new DistributedTask<Result>(new CheckAffinityPolicy(content), member);
		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		try {
			// gets result
			result = task.get();
		} catch (Exception e) {
			throw new Exception(e.getMessage(), e);
		}
		return result;
	}

	/**
	 * Returns the configuration file for the environment
	 * 
	 * @param node node where execute a future task to get the config file 
	 * @param what type of configuration file to return
	 * @return Configuration file container
	 * @throws Exception if exception occurs
	 */
	public ConfigurationFile getEnvConfigFile(String what) throws Exception{
		// checks if the user is authorized to read configuration
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_CLUSTER_CONFIGURATION));

		ConfigurationFile result = null;

		Member member = getMember();

		// creates the future task
		DistributedTask<ConfigurationFile> task = null;

		// checks if wants Hazelcast or JEM node configuration
		// by default is JEM node configuration
		if (what != null){
			if (what.equalsIgnoreCase(ConfigKeys.HAZELCAST_CONFIG)){
				task = new DistributedTask<ConfigurationFile>(new GetHazelcastConfiguration(), member);
			} else if (what.equalsIgnoreCase(ConfigKeys.JEM_ENV_CONF)){
				task = new DistributedTask<ConfigurationFile>(new GetJemEnvConfiguration(), member);
			}
		} else {
			task = new DistributedTask<ConfigurationFile>(new GetJemEnvConfiguration(), member);
		}

		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		try {
			// gets result
			result = task.get();
		} catch (Exception e) {
			throw new Exception(e.getMessage(), e);
		}
		return result;
	}
	
	/**
	 * Returns the configuration file for the environment
	 * 
	 * @param file configuration file to save 
	 * @param what type of configuration file to return
	 * @return Configuration file container
	 * @throws Exception if exception occurs
	 */
	public ConfigurationFile saveEnvConfigFile(ConfigurationFile file, String what) throws Exception{
		// checks if the user is authorized to read configuration
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_CLUSTER_CONFIGURATION));

		ConfigurationFile result = null;

		Member member = getMember();

		// creates the future task
		DistributedTask<ConfigurationFile> task = null;

		// checks if wants Hazelcast or JEM node configuration
		// by default is JEM node configuration
		if (what != null){
			if (what.equalsIgnoreCase(ConfigKeys.HAZELCAST_CONFIG)){
				checkConfigFile(file.getContent(), what);
				task = new DistributedTask<ConfigurationFile>(new SaveHazelcastConfiguration(file), member);
			} else if (what.equalsIgnoreCase(ConfigKeys.JEM_ENV_CONF)){
				checkConfigFile(file.getContent(), what);
				task = new DistributedTask<ConfigurationFile>(new SaveJemEnvConfiguration(file), member);
			}
		} else {
			task = new DistributedTask<ConfigurationFile>(new SaveJemEnvConfiguration(file), member);
		}
		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		try {
			// gets result
			result = task.get();
		} catch (Exception e) {
			throw new Exception(e.getMessage(), e);
		}
		return result;
	}
	
	/**
	 * Returns the top command result
	 * 
	 * @param node node where execute a future task to get top command 
	 * @return content file in String
	 * @throws Exception if exception occurs
	 */

	public String top(NodeInfoBean node) throws Exception{
		// checks if the user is authorized to performs commands
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_COMMANDS));
		String result = null;
		// gets nodes map instance 
		IMap<String, NodeInfo> members_map = getInstance().getMap(Queues.NODES_MAP);
		// is not a super node and is not UNKNOWN
		if (!node.isSuperNode() && !node.getStatus().equalsIgnoreCase(Status.UNKNOWN.getDescription())){
			// gets key
			String key = node.getKey();
			// checks if is in map
			if (members_map.containsKey(key)){
				// if is DRAINED or DRAINING than starts it 
				// gets the cluster to have member object of Hazelcast
				// to execute the future task
				Cluster cluster = getInstance().getCluster();
				// gets all members and scans them
				Set<Member> set = cluster.getMembers();
				for (Member member : set){
					String memberKey = member.getUuid();
					// is the same member
					if (node.getKey().equalsIgnoreCase(memberKey)){
						// creates the future task
						DistributedTask<String> task = new DistributedTask<String>(new Top(), member);
						// gets executor service and executes!
						ExecutorService executorService = getInstance().getExecutorService();
						executorService.execute(task);
						try {
							result = task.get();
						} catch (Exception e) {
							throw new Exception(e.getMessage(), e);
						}
					} 
				}
			}
		}
		return result;
	}

	/**
	 * Returns part of JEM node log
	 * 
	 * @param node node where execute a future task to get top command 
	 * @return content file in String
	 * @throws Exception if exception occurs
	 */
	public String log(NodeInfoBean node) throws Exception{
		// checks if the user is authorized to performs commands
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_COMMANDS));
		
		String result = null;
		// gets nodes map instance 
		IMap<String, NodeInfo> members_map = getInstance().getMap(Queues.NODES_MAP);
		// is not a super node and is not UNKNOWN
		if (!node.isSuperNode() && !node.getStatus().equalsIgnoreCase(Status.UNKNOWN.getDescription())){
			// gets key
			String key = node.getKey();
			// checks if is in map
			if (members_map.containsKey(key)){
				// if is DRAINED or DRAINING than starts it 
				// gets the cluster to have member object of Hazelcast
				// to execute the future task
				Cluster cluster = getInstance().getCluster();
				// gets all members and scans them
				Set<Member> set = cluster.getMembers();
				for (Member member : set){
					String memberKey = member.getUuid();
					// is the same member
					if (node.getKey().equalsIgnoreCase(memberKey)){
						// creates the future task
						DistributedTask<String> task = new DistributedTask<String>(new GetLog(), member);
						// gets executor service and executes!
						ExecutorService executorService = getInstance().getExecutorService();
						executorService.execute(task);
						try {
							// get resuls
							result = task.get();
						} catch (Exception e) {
							throw new Exception(e.getMessage(), e);
						}
					} 
				}
			}
		}
		return result;
	}

    
    /**
	 * Returns the HAZELCAST cluster status which is the list of all members.<br>
	 * This is a sampl output format:<br>
	 * <code>
	 *  Members [2] {
    	    Member [127.0.0.1]:5710 this
    	    Member [127.0.0.1]:5711 lite
    	}
	 * </code>
	 * 
	 * @param node node where execute a future task to get top command 
	 * @return content file in String
	 * @throws Exception if exception occurs
     */
    public String displayCluster(NodeInfoBean node) throws Exception{
		// checks if the user is authorized to performs commands
		// if not, this method throws an exception
    	checkAuthorization(new StringPermission(Permissions.ADMINISTRATION_NODES_COMMANDS));

    	// gets hzelcast cluster
    	Cluster cluster =  getInstance().getCluster();
    	
    	// scans all members creating a stringbuffer
    	// with HC format
    	StringBuffer result = new StringBuffer("Members [").append(cluster.getMembers().size()).append("] {").append("\n");
    	for (Member member : cluster.getMembers()){
    		// adds dinamically the label "this"
    		// based on node passed as argument
    		String memberString = StringUtils.remove(member.toString(), " this");
    		if (node.getKey().equals(member.getUuid())){
    			memberString = memberString + " this";
    		}
    		result.append("    ").append(memberString).append("\n");
    	}
    	result.append("}\n");
    	return result.toString();
    }
	
}