/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.services;

import java.util.Collection;
import java.util.Date;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.apache.shiro.SecurityUtils;
import org.apache.shiro.subject.Subject;
import org.pepstock.jem.gwt.server.UserInterfaceMessage;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.executors.certificates.GetCryptedValueAndHash;
import org.pepstock.jem.node.resources.CryptedValueAndHash;
import org.pepstock.jem.node.resources.Resource;
import org.pepstock.jem.node.security.Permissions;
import org.pepstock.jem.node.security.StringPermission;
import org.pepstock.jem.node.security.User;
import org.pepstock.jem.util.filters.Filter;
import org.pepstock.jem.util.filters.FilterToken;
import org.pepstock.jem.util.filters.fields.ResourceFilterFields;
import org.pepstock.jem.util.filters.predicates.ResourcePredicate;

import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;

/**
 * This service manages all common resources inside of JEM.
 * 
 * @author Andrea "Stock" Stocchero
 *
 */
public class CommonResourcesManager extends DefaultService{

	/**
	 * Puts a resource in JEM, adding if new or replacing if exists.<br>
	 * Checks if the user is authorized to add and update resources
	 * 
	 * @param resource resource to add
	 * @return <code>true</code> if the resource has been added otherwise <code>false</code> if replaced
	 * @throws Exception if any exception occurs
	 */
	public boolean put(Resource resource) throws Exception {
		// checks user authentication
		// if not, this method throws an exception
		checkAuthentication();
		
		// checks if the key (resource name) exists
		IMap<String, Resource> map = getInstance().getMap(Queues.COMMON_RESOURCES_MAP);
		if (map.containsKey(resource.getName())){
			// checks if the user is authorized to update resource
			// if not, this method throws an exception
			checkAuthorization(new StringPermission(Permissions.RESOURCES_UPDATE));
			try {
				// locks the key 
				map.lock(resource.getName());
				
				// gets old object and checks user
				// this is necessary to check if the new object 
				// is the same and none has updated it in the meantime
				Resource oldResource = map.get(resource.getName());
				if (oldResource.getUser() != null){
					// checks if the user is the same
					// if not, throws an exception.
					// that means the someone else changed the object
					if (!oldResource.getUser().equalsIgnoreCase(resource.getUser())){
						LogAppl.getInstance().emit(UserInterfaceMessage.JEMG047E, oldResource, resource);
						throw new Exception(UserInterfaceMessage.JEMG047E.toMessage().getFormattedMessage(oldResource, resource));
					}
				}
				// checks last modified
				// this is necessary to check if the new object 
				// is the same and none has updated it in the meantime
				if (oldResource.getLastModified() != null){
					// checks if the same
					// if not, throws an exception.
					// that means the someone else changed the object
					if (!oldResource.getLastModified().equals(resource.getLastModified())){
						LogAppl.getInstance().emit(UserInterfaceMessage.JEMG047E, oldResource, resource);
						throw new Exception(UserInterfaceMessage.JEMG047E.toMessage().getFormattedMessage(oldResource, resource));
					}
				}
				
				// here the update is consistent so
				// gets user info and time storing that on
				// object
				Subject currentUser = SecurityUtils.getSubject();
				User userPrincipal = (User)currentUser.getPrincipal();
				String userId = userPrincipal.getId();
				resource.setUser(userId);
				resource.setLastModified(new Date());
				// replaces on map
				map.replace(resource.getName(), resource);
			} finally {
				// unlocks always the key
				map.unlock(resource.getName());
			}
			// returns false because it updates
			// a key already present in map
			return false;
		} else {
			// checks if the user is authorized to create resources
			// if not, this method throws an exception
			checkAuthorization(new StringPermission(Permissions.RESOURCES_CREATE));
			try {
				// locks the key 
				map.lock(resource.getName());
				
				// gets user info and time storing that on
				// object
				Subject currentUser = SecurityUtils.getSubject();
				User userPrincipal = (User)currentUser.getPrincipal();
				String userId = userPrincipal.getId();
				resource.setUser(userId);
				resource.setLastModified(new Date());
				// puts on map
				map.put(resource.getName(), resource);
			} finally {
				// unlocks always the key
				map.unlock(resource.getName());
			}
			// returns true because it creates
			// new object
			return true;
		}
	}

	/**
	 * Gets the resource by name
	 * 
	 * @param name name of resource
	 * @return resource instance
	 * @throws Exception if any exception occurs
	 */
	public Resource get(String name) throws Exception {
		// checks if the user is authorized to read resources information
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.RESOURCES_READ));
		IMap<String, Resource> map = getInstance().getMap(Queues.COMMON_RESOURCES_MAP);
		// checks if the key (resource name) exists
		if (map.containsKey(name)){
			Resource resource = null;
			try {
				// locks the key 
				map.lock(name);
				// gets the resource 
				// to return
				resource = map.get(name);
			} finally {
				// unlocks always the key
				map.unlock(name);
			}
			return resource;
		}
		// resource is not in map. Returns null
		return null;
	}

	/**
	 * Removes a resource by name.
	 * 
	 * @param name name of resource
	 * @return <code>true</code> if the resource has been removed otherwise <code>false</code> if not
	 * @throws Exception if any exception occurs
	 */
	public boolean remove(String name) throws Exception {
		// checks if the user is authorized to delete resources
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.RESOURCES_DELETE));
		IMap<String, Resource> map = getInstance().getMap(Queues.COMMON_RESOURCES_MAP);
		// checks if the key (resource name) exists
		if (map.containsKey(name)){
			Resource resource = null;
			try {
				// locks the key 
				map.lock(name);
				// gets the resource removing it 
				// from map 
				resource = map.remove(name);
			} finally {
				// unlocks always the key
				map.unlock(name);
			}
			// returns true if object was in the map
			// otherwise false
			return resource != null;
		}
		// key is not in map 
		// then return false
		return false;
	}

	/**
	 * Gets all resources list defined in JEM
	 * 
	 * @param filter filter of resources
	 * @return collection of resources
	 * @throws Exception if any exception occurs
	 */
	public Collection<Resource> values(String filter) throws Exception {
		// checks if the user is authorized to read resources information
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.RESOURCES_READ));
		
		IMap<String, Resource> map = getInstance().getMap(Queues.COMMON_RESOURCES_MAP);
		// creates a Resource predicate 
		// using filter filled on UI 
		ResourcePredicate predicate;
		try {
			 predicate = new ResourcePredicate(Filter.parse(filter));
		} catch (Exception e) {
			// default case, all resources with empty filter by name
			Filter all = new Filter();
			all.add(new FilterToken(ResourceFilterFields.NAME.getName(), StringUtils.EMPTY));
			predicate = new ResourcePredicate(all);
		}
		
		Collection<Resource> result = null;
		// locks all map to have a consistent collection
		// only for 10 seconds otherwise 
		// throws an exception
		if (map.lockMap(10, TimeUnit.SECONDS)){ 
			try {
				// performs predicate to have the collection
				result = map.values(predicate);
			} finally {
				// unlocks always the map
				map.unlockMap();
			}
		} else {
			// timeout exception
			throw new Exception(UserInterfaceMessage.JEMG022E.toMessage().getFormattedMessage(Queues.COMMON_RESOURCES_MAP));
		}
		// returns a collection
		return result;

	}
	
	/**
	 * Returns a container of a secret, crypted and hashed.<br>
	 * This uses a key automatically creates for whole cluster.<br>
	 * This is necessary to crypt possible password when you define new resources
	 * 
	 * @param secret secret value to crypt and hash
	 * @return a container of crypted and hash
	 * @throws Exception if any exception occurs
	 */
	public CryptedValueAndHash getEncryptedSecret(String secret) throws Exception {
		// checks if the user is authorized to create resources
		// if not, this method throws an exception
		checkAuthorization(new StringPermission(Permissions.RESOURCES_CREATE));
		
		// gets Hazelcast member
		// if is not able, an exception occurs
		Member member = getMember();
		// creates distributed task to crypt and hash secret
		DistributedTask<CryptedValueAndHash> task = new DistributedTask<CryptedValueAndHash>(new GetCryptedValueAndHash(secret), member);
		ExecutorService executorService = getInstance().getExecutorService();
		// executes it
		executorService.execute(task);
		CryptedValueAndHash crypt;
		try {
			// gets result
			crypt = task.get();
		} catch (InterruptedException e) {
			throw new Exception(e.getMessage(), e);
		} catch (ExecutionException e) {
			throw new Exception(e.getMessage(), e);
		}
		// return result
		return crypt;
	}
	
}