package org.pepstock.jem.gwt.server.services;

import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.ExecutorService;

import org.pepstock.jem.node.executors.resources.GetResourceDefinitions;
import org.pepstock.jem.node.executors.resources.GetResourceDescriptor;
import org.pepstock.jem.node.executors.resources.GetResourceNames;
import org.pepstock.jem.node.resources.custom.ResourceDefinition;
import org.pepstock.jem.node.resources.custom.ResourceDescriptor;
import org.pepstock.jem.node.security.Permissions;
import org.pepstock.jem.node.security.StringPermission;

import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.Member;

/**
 * This service manages all common custom resource definitions inside JEM.
 *  
 * @author Marco "Fuzzo" Cuccato
 */
public class CommonResourceDefinitionsManager extends DefaultService {

	/**
	 * Gets all custom {@link ResourceDefinition}s.
	 * @return a collection of {@link ResourceDefinition}
	 * @throws Exception
	 */
	public Collection<ResourceDefinition> getAllDefinitions() throws Exception {
		// checks if the user is authorized to search job
		// if not, this method throws an exception
		String permission = Permissions.RESOURCES_READ;
		checkAuthorization(new StringPermission(permission));

		// gets member of Hazelcast where executes executor
		// if is not able to have it, throws an exception
		Member member = getMember();
		// creates executor
		DistributedTask<Collection<ResourceDefinition>> task = new DistributedTask<Collection<ResourceDefinition>>(new GetResourceDefinitions(), member);
		ExecutorService executorService = getInstance().getExecutorService();
		// executes it
		executorService.execute(task);
		// gets result
		return task.get();
	}

	/**
	 * Gets all custom resource names. Is intended to be used by GUI to build new resource selector.
	 * @return a collection of String
	 * @throws Exception
	 */
	public Collection<String> getAllResourceNames() throws Exception {
		String permission = Permissions.RESOURCES_READ;
		checkAuthorization(new StringPermission(permission));
		
		Member member = getMember();
		DistributedTask<Collection<String>> task = new DistributedTask<Collection<String>>(new GetResourceNames(), member);
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		return task.get();
	}
	
	/**
	 * Get the {@link ResourceDescriptor} identified by <code>resourceName</code>
	 * @param resourceName the resource name
	 * @return the resource descriptor
	 * @throws Exception
	 */
	public ResourceDescriptor getDescriptorOf(String resourceName) throws Exception {
		String permission = Permissions.RESOURCES_READ;
		checkAuthorization(new StringPermission(permission));
		
		Member member = getMember();
		DistributedTask<ResourceDescriptor> task = new DistributedTask<ResourceDescriptor>(new GetResourceDescriptor(resourceName), member);
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		return task.get();
		
	}

	/**
	 * Gets all custom common resource descriptors. Is intended to be used by GUI to builds panel.
	 * @return a collection of {@link ResourceDescriptor}
	 * @throws Exception
	 */
	public Collection<ResourceDescriptor> getAllDescriptors() throws Exception {
		String permission = Permissions.RESOURCES_READ;
		checkAuthorization(new StringPermission(permission));

		// security checks performed by getAllDefinitions(), skipped
		Collection<ResourceDefinition> content = getAllDefinitions();	
		// builda ResourceDescriptor collection
		Collection<ResourceDescriptor> toReturn = new ArrayList<ResourceDescriptor>();
		if (content != null) {
			for (ResourceDefinition rd : content) {
				toReturn.add(rd.getResourceDescriptor());
			}
		}
		return toReturn;
	}

}
