/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.services;

import java.util.Collection;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;

import org.apache.commons.lang3.StringUtils;
import org.pepstock.jem.node.executors.certificates.AddCertificate;
import org.pepstock.jem.node.executors.certificates.GetCertificates;
import org.pepstock.jem.node.executors.certificates.RemoveCertificate;
import org.pepstock.jem.node.security.CertificateEntry;
import org.pepstock.jem.node.security.Permissions;
import org.pepstock.jem.node.security.StringPermission;

import com.hazelcast.core.DistributedTask;
import com.hazelcast.core.Member;

/**
 * This service manages all roles and their authorizations and users relations inside of JEM.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class CertificatesManager extends DefaultService{

	/**
	 * Returns a collection of certificates, by a filter (a set of key-values).
	 * 
     * @param filter string which contains a set of key-values
     * @return a collection of certificates, matching the filter
     * @throws Exception if any exception occurs
     */
    public Collection<CertificateEntry> getCertificates(String filter) throws Exception {
		// checks if the user is authorized to read aliases of certificate
		// if not, this method throws an exception
    	checkAuthorization(new StringPermission(Permissions.CERTIFICATES_READ));
    	if (filter.equals("*")){
    		filter = "";
    	} else if (filter.endsWith("*")){
    		filter = StringUtils.substringBeforeLast(filter, "*");
    	} 
    	Member member = getMember();

		// creates cancel executor
		DistributedTask<Collection<CertificateEntry>> task = new DistributedTask<Collection<CertificateEntry>>(new GetCertificates(filter), member);
		ExecutorService executorService = getInstance().getExecutorService();
		// executes it
		executorService.execute(task);

		Collection<CertificateEntry> content;
		try {
			// gets result
			content = task.get();
		} catch (InterruptedException e) {
			throw new Exception(e.getMessage(), e);
		} catch (ExecutionException e) {
			throw new Exception(e.getMessage(), e);
		}
		return content;

    }


    /**
     * Adds a new certificate in JEM.
     * 
     * @param certificate 
     * @param alias 
     * @return always true
     * @throws Exception if any exception occurs
     */
    public Boolean addCertificate(byte[] certificate, String alias) throws Exception {
		// checks if the user is authorized to create certificates
		// if not, this method throws an exception    	
    	checkAuthorization(new StringPermission(Permissions.CERTIFICATES_CREATE));
    	Member member = getMember();

		DistributedTask<Boolean> task = new DistributedTask<Boolean>(new AddCertificate(certificate, alias), member);
		// gets executor service and executes!
		ExecutorService executorService = getInstance().getExecutorService();
		executorService.execute(task);
		
		try {
			// gets result
			task.get();
		} catch (InterruptedException e) {
			throw new Exception(e.getMessage(), e);
		} catch (ExecutionException e) {
			throw new Exception(e.getMessage(), e);
		}
    	return Boolean.TRUE;
    }

    /**
     * Removes a collection of certificates from JEM.
     * 
     * @param entries collection of certificates to be removed 
     * @return always true
     * @throws Exception if any exception
     */
    public Boolean removeCertificate(Collection<CertificateEntry> entries) throws Exception {
		// checks if the user is authorized to delete certificates
		// if not, this method throws an exception
    	checkAuthorization(new StringPermission(Permissions.CERTIFICATES_DELETE));
    	
    	Member member = getMember();
    	
		// scans all certificates
		for (CertificateEntry entry : entries){
			// creates the future task
			DistributedTask<Boolean> task = new DistributedTask<Boolean>(new RemoveCertificate(entry.getAlias()), member);
			// gets executor service and executes!
			ExecutorService executorService = getInstance().getExecutorService();
			executorService.execute(task);
			
			try {
				// gets result
				task.get();
			} catch (InterruptedException e) {
				throw new Exception(e.getMessage(), e);
			} catch (ExecutionException e) {
				throw new Exception(e.getMessage(), e);
			}
		}
    	
    	return Boolean.TRUE;
    }

}