/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.security;

import java.util.Collection;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.shiro.cache.Cache;
import org.apache.shiro.cache.CacheException;
import org.pepstock.jem.gwt.server.UserInterfaceMessage;
import org.pepstock.jem.gwt.server.commons.SharedObjects;
import org.pepstock.jem.log.LogAppl;

import com.hazelcast.core.Cluster;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.hazelcast.core.MembershipEvent;
import com.hazelcast.core.MembershipListener;

/**
 * Cache used by SHIRO for authorizations. This uses a normal HashMap in memeory if is not able to use Jem group, by hazelcast.
 * If the group is available, uses Hazelcast.<br>
 * Is a Hazelcast listener to understand when to use Hazeclast instead of memory.<br>
 * Blocking the access to web site if JEM group is down, it should never use memory hashmap. 
 * 
 * @author Andrea "Stock" Stocchero
 * @param <K> 
 * @param <V> 
 *
 */
public class JemCache<K, V> implements Cache<K, V>, MembershipListener {

	private String name = null;
	
	private ConcurrentMap<K, V> map = null;
	
	private boolean memory = true;
	
	/**
	 * Stored the name of cache and checks if hazelcast is available to use it. Otherwise a memory instance will be used.
	 * 
	 * @param name chace map name
	 * 
	 */
    public JemCache(String name) {
		setName(name);
		// gets the cluster and add a listener so can know if cluster has members or not
		Cluster cluster = SharedObjects.getInstance().getLocalMember().getCluster();
		cluster.addMembershipListener(this);
		
		// checks if hazeclast is available
		if (SharedObjects.getInstance().isDataClusterAvailable()){
			setHazelcastCache();
		} else {
			setMemoryCache();
		}
	}

	/**
	 * @return the cache name
	 */
	public String getName() {
		return name;
	}

	/**
	 * @param name the cache name to set
	 */
	public void setName(String name) {
		this.name = name;
	}

	/**
	 * @return the memory
	 */
	public boolean isMemory() {
		return memory;
	}

	/**
	 * @param memory the memory to set
	 */
	public void setMemory(boolean memory) {
		this.memory = memory;
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#clear()
	 */
	@Override
	public void clear() throws CacheException {
		map.clear();
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#get(java.lang.Object)
	 */
	@Override
	public V get(K arg0) throws CacheException {
		return map.get(arg0);
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#keys()
	 */
	@Override
	public Set<K> keys() {
		return map.keySet();
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#put(java.lang.Object, java.lang.Object)
	 */
	@Override
	public V put(K arg0, V arg1) throws CacheException {
		return map.put(arg0, arg1);
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#remove(java.lang.Object)
	 */
	@Override
	public V remove(K arg0) throws CacheException {
		return map.remove(arg0);
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#size()
	 */
	@Override
	public int size() {
		return map.size();
	}

	/* (non-Javadoc)
	 * @see org.apache.shiro.cache.Cache#values()
	 */
	@Override
	public Collection<V> values() {
		return map.values();
	}
	
	
	/* (non-Javadoc)
	 * @see com.hazelcast.core.MembershipListener#memberAdded(com.hazelcast.core.MembershipEvent)
	 */
    @Override
    public void memberAdded(MembershipEvent event) {
		Member member = event.getMember();
		// if group is already available, checks if memeory is set
		if (!SharedObjects.getInstance().isDataClusterAvailable()){
			// if is lite, do nothing because it doesn't change anything
			if (!member.isLiteMember()){
				// we have a new member of cluster!!!
				// set HAZELCAST
				setHazelcastCache();
			}
		} else {
			if (isMemory()){
				// set HAZECLAST because before was set MEMORY 
				// but now the JEM group is available
				setHazelcastCache();
			}
		}
    }

	/* (non-Javadoc)
	 * @see com.hazelcast.core.MembershipListener#memberRemoved(com.hazelcast.core.MembershipEvent)
	 */
    @Override
    public void memberRemoved(MembershipEvent event) {
    	// gets removed member
		Member memberRemoved = event.getMember();
		// if removed member is lite, do nothing 
		if (memberRemoved.isLiteMember())
			return;
		
		// scans all member to look if an active member is running
		Cluster cluster = SharedObjects.getInstance().getLocalMember().getCluster();
		Set<Member> setMembers  = cluster.getMembers();
		for (Member member : setMembers) {
			if (!member.isLiteMember()){
				// there is at least a member so JEM group member is available 
				return;
			}
		}
		// scan is finished so no member is available!!
		// so activate memory cache
		setMemoryCache();
    }

    /**
     * Sets HAZECALST cache. Uses the cache name asked by SHIRO
     */
    @SuppressWarnings("unchecked")
    private void setHazelcastCache(){
    	LogAppl.getInstance().emit(UserInterfaceMessage.JEMG020I, "HAZELCAST");
		HazelcastInstance instance = SharedObjects.getInstance().getLocalMember();
		map = (IMap<K, V>) instance.getMap(getName());
		setMemory(false);
    }
    
    /**
     * Sets Memory cache!  
     */
    private void setMemoryCache(){
    	LogAppl.getInstance().emit(UserInterfaceMessage.JEMG020I, "MEMORY");
		map = new ConcurrentHashMap<K, V>();
		setMemory(true);
    }

}