/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.listeners;

import java.io.FileNotFoundException;
import java.util.Set;

import javax.servlet.ServletContext;
import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;

import org.apache.shiro.web.env.EnvironmentLoaderListener;
import org.pepstock.jem.gwt.server.UserInterfaceMessage;
import org.pepstock.jem.gwt.server.commons.NodeInfoUtility;
import org.pepstock.jem.gwt.server.commons.SharedObjects;
import org.pepstock.jem.gwt.server.configuration.ConfigurationException;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeInfo;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.Status;
import org.pepstock.jem.node.configuration.ConfigKeys;

import com.hazelcast.config.FileSystemXmlConfig;
import com.hazelcast.core.Cluster;
import com.hazelcast.core.Hazelcast;
import com.hazelcast.core.HazelcastInstance;
import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;

/**
 * Context listener which starts up JEM web application, initializing Log4j, SharedObject and Hazelcast.<br>
 * Extends SHIRO startup, reading the context attribute <code>hazelcast.config</code> which contains the configuration
 * file to start Hazelcast.
 * 
 * @author Andrea "Stock" Stocchero
 *
 */
public class StartUp extends EnvironmentLoaderListener implements ServletContextListener {
	
	/**
	 * Constraucts initializing log4j and shared objects
	 */
	public StartUp() {
		// starts log4j
		LogAppl.getInstance();
		// creates shared objects
		SharedObjects.createInstance();
		// sets IPV4
		System.setProperty(ConfigKeys.IPV4, Boolean.TRUE.toString());
	}

	/* (non-Javadoc)
	 * @see javax.servlet.ServletContextListener#contextDestroyed(javax.servlet.ServletContextEvent)
	 */
	@Override
	public void contextDestroyed(ServletContextEvent event){
		// stop swarm instance
		SharedObjects.getInstance().getMainSwarm().shutDown();
		// stop client instance
		SharedObjects.getInstance().shutdownLocalMember();
		// calls the super class
		super.contextDestroyed(event);
		
	}
	/* (non-Javadoc)
	 * @see javax.servlet.ServletContextListener#contextInitialized(javax.servlet.ServletContextEvent)
	 */
	@Override
	public void contextInitialized(ServletContextEvent event) {
		// gets servlet context
		ServletContext context = event.getServletContext();
    	// gets servlet real path
		String contextPath = context.getRealPath(".");
		contextPath = contextPath.substring(0, contextPath.length() - 1);

		// reads Hazecast init parameter
		String hazelcastFile = context.getInitParameter(ConfigKeys.HAZELCAST_CONFIG);
		// set the check version flag retreive by web.xml
		Boolean checkVersion = Boolean.valueOf(context.getInitParameter(ConfigKeys.JEM_CHECK_VERSION));
		SharedObjects.getInstance().setCheckVersion(checkVersion);
        if (hazelcastFile != null) {
    		try {
    			// starts hazelcast
	            startHazelcast(contextPath, hazelcastFile);
            } catch (ConfigurationException e) {
	            throw new RuntimeException(e);
            }
        } else {
        	throw new RuntimeException(UserInterfaceMessage.JEMG021E.toMessage().getFormattedMessage());
        }
        //starts SHIRO
		super.contextInitialized(event);
	}

	/**
	 * Starts Hazelcast member in lite mode.
	 * 
	 * @param filename Hazelast config file
	 * @throws ConfigurationException if a config error occurs
	 */
	private void startHazelcast(String contextPath, String hazelcastFile) throws ConfigurationException{
		LogAppl.getInstance().emit(UserInterfaceMessage.JEMG006I);
		String filename=contextPath+hazelcastFile;
		// loads configuration file from XML file
		FileSystemXmlConfig config;
        try {
	        config = new FileSystemXmlConfig(filename);
        } catch (FileNotFoundException e) {
	        throw new ConfigurationException(e);
        }
        // sets it as lite member
		config.setLiteMember(true);
		
		// starts Hazelcast
		HazelcastInstance instance = Hazelcast.newHazelcastInstance(config);
		SharedObjects.getInstance().setLocalMember(instance);
		// saves teh instance in a hashmap (NOT USED at the moment)
		//String group = instance.getConfig().getGroupConfig().getName();
		//SharedObjects.getInstance().getGroupsContainer().put(group, instance);
		
		// gets teh cluster and add a listener so can know if cluster has members or not
		Cluster cluster = instance.getCluster();
		cluster.addMembershipListener(new NodeListener());

		Member currentMember = cluster.getLocalMember();
		String key = currentMember.getUuid();

		// checks is cluster is available
		if (isDataAvailable(cluster)){
			IMap<String, NodeInfo> members_map = instance.getMap(Queues.NODES_MAP);
			// node is already there (could happen if the node was down)
			if (members_map.containsKey(key)){
				try{
					members_map.lock(key);
					// gets node  
					NodeInfo info = members_map.get(key);
					//sets status and supernode true
					info.setStatus(Status.ACTIVE);
					info.setSuperNode(true);
					// saves object
					SharedObjects.getInstance().setInfo(info);
					// replaces in central map
					members_map.replace(key, info);
					LogAppl.getInstance().emit(UserInterfaceMessage.JEMG007I, info.getKey());
				} finally{
					members_map.unlock(key);
				}
			} else {
				// is not in map so creates new node info 
				NodeInfo info = NodeInfoUtility.createNodeInfo(currentMember, contextPath);
				//sets status and supernode true
				info.setSuperNode(true);
				info.setStatus(Status.ACTIVE);
				// saves in shared object
				SharedObjects.getInstance().setInfo(info);
				// stores into shared map
				NodeInfoUtility.checkAndStoreNodeInfo(info);
				LogAppl.getInstance().emit(UserInterfaceMessage.JEMG007I, info.getKey());
			}
		} else {
			// creates only the node and stores only in cache of this jvm
			NodeInfo info = NodeInfoUtility.createNodeInfo(currentMember, contextPath);
			info.setSuperNode(true);
			info.setStatus(Status.ACTIVE);
			SharedObjects.getInstance().setInfo(info);
			LogAppl.getInstance().emit(UserInterfaceMessage.JEMG007I, info.getKey());
		}
	}

	/**
	 * Checks if cluster has active members which can provide data
	 * @param cluster cluster of JEM
	 * @return <code>true</code> is cluster has active members, otherwise <code>false</code> 
	 */
	private boolean isDataAvailable(Cluster cluster){
		Set<Member> members  = cluster.getMembers();
		// if not empty (shouldn't be empty)
		if (!members.isEmpty()){
			Member member = cluster.getMembers().iterator().next();
			// at first NO LITE MEMBERS means the cluster has active members
			if (!member.isLiteMember()){
				SharedObjects.getInstance().setDataClusterAvailable(true);
				return true;
			}
		}
		// sets to false
		SharedObjects.getInstance().setDataClusterAvailable(false);
		return false;
	}

}