/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.commons;

import org.pepstock.jem.gwt.server.UserInterfaceMessage;
import org.pepstock.jem.gwt.server.swarm.MainSwarm;
import org.pepstock.jem.gwt.server.swarm.SwarmNodeMessage;
import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.NodeInfo;

import com.hazelcast.core.HazelcastInstance;

/**
 * Container of shared (and then static) references to use every where.<br>
 * It contains node info this node, local member of Hazelcast and a map of
 * hazelcast instances (not used yet).
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class SharedObjects {

	private static SharedObjects SHARED_OBJECTS = null;

	private NodeInfo info = null;

	private boolean isDataClusterAvailable = false;

	private HazelcastInstance localMember = null;

	private String sessionsInstance = null;

	private MainSwarm main = null;

	private Boolean checkVersion = false;

	private String jemVersion = NodeInfo.UNKNOWN_VERSION;

	/**
	 * Creates a new object, if doesn't exists.
	 * 
	 * @return the shared instance
	 */
	public static synchronized SharedObjects createInstance() {
		if (SHARED_OBJECTS == null) {
			SHARED_OBJECTS = new SharedObjects();
		}
		return SHARED_OBJECTS;
	}

	/**
	 * Returns the shared instance.
	 * 
	 * @return the shared instance
	 */
	public static SharedObjects getInstance() {
		return SHARED_OBJECTS;
	}

	/**
	 * Empty constructor which creates a main swarm engine
	 */
	private SharedObjects() {
		main = new MainSwarm();
	}

	/**
	 * Sets node info object
	 * 
	 * @param info
	 *            node info instance
	 */
	public void setInfo(NodeInfo info) {
		this.info = info;
	}

	/**
	 * Returns node info object
	 * 
	 * @return node info
	 */
	public NodeInfo getInfo() {
		return info;
	}

	/**
	 * Sets <code>true</code> if hazelcast cluster has node with data.
	 * 
	 * @param isDataClusterAvailable
	 *            <code>true</code> if hazelcast cluster has node with data
	 */
	public void setDataClusterAvailable(boolean isDataClusterAvailable) {

		// sets check if SWARM must be started
		// based on current status of JEM group
		boolean mustBeStarted = (isDataClusterAvailable)
				&& (!this.isDataClusterAvailable);
		boolean mustBeShuttedDown = (!isDataClusterAvailable)
				&& (this.isDataClusterAvailable);

		// sets current JEM group available
		this.isDataClusterAvailable = isDataClusterAvailable;
		if (isDataClusterAvailable) {
			LogAppl.getInstance().emit(UserInterfaceMessage.JEMG004I, SharedObjects.getInstance().getLocalMember().getConfig().getGroupConfig().getName());
			// if SWARM must be started
			if (mustBeStarted) {
				try {
					// starts
					main.start();
				} catch (Exception e) {
					LogAppl.getInstance().emit(SwarmNodeMessage.JEMO015E, e);
				}
			}
			// if SWARM must be stopped
			if (mustBeShuttedDown) {
				try {
					// stops
					main.shutDown();
				} catch (Exception e) {
					LogAppl.getInstance().emit(SwarmNodeMessage.JEMO015E, e);
				}
			}
		} else {
			LogAppl.getInstance().emit(UserInterfaceMessage.JEMG005E, SharedObjects.getInstance().getLocalMember().getConfig().getGroupConfig().getName());
		}
	}

	/**
	 * Returns if web node is connected to existing JEM cluster.
	 * @return <code>true</code> if Hazelcast cluster has node with data
	 */
	public boolean isDataClusterAvailable() {
		return isDataClusterAvailable;
	}

	/**
	 * Shutdown the Hazelcast member
	 */
	public void shutdownLocalMember() {
		if (localMember != null) {
			localMember.getLifecycleService().shutdown();
			localMember = null;
		}
	}

	/**
	 * Returns Hazelcast local instance
	 * 
	 * @return the localMember of Hazelcast
	 */
	public HazelcastInstance getLocalMember() {
		return localMember;
	}

	/**
	 * Sets Hazelcast local instance
	 * @param localMember
	 *            the Hazelcast localMember to set
	 */
	public void setLocalMember(HazelcastInstance localMember) {
		this.localMember = localMember;
	}

	/**
	 * Returns session instance name
	 * @return the sessionsInstance name of session sharing, used by SHIRO to
	 *         cache authorization
	 */
	public String getSessionsInstance() {
		return sessionsInstance;
	}

	/**
	 * Sets session instance name
	 * @param sessionsInstance
	 *            the sessionsInstance to set, name of session sharing, used by
	 *            SHIRo to cache authorization
	 */
	public void setSessionsInstance(String sessionsInstance) {
		this.sessionsInstance = sessionsInstance;
	}

	/**
	 * Returns main swarm engine
	 * 
	 * @return the main swarm engine
	 */
	public MainSwarm getMainSwarm() {
		return main;
	}

	/**
	 * Returns if node must check the version of JEM
	 * @return the checkVersion
	 *         <p>
	 *         if true the system will check the JEM WEB version against the JEM
	 *         NODES version and if there is a mismatch the system will exit
	 *         with error. If false the system will do the chek and in case of
	 *         failure set a warning
	 */
	public Boolean getCheckVersion() {
		return checkVersion;
	}

	/**
	 * Sets if node must check the version of JEM
	 * @param checkVersion
	 *            the checkVersion to set.
	 *            <p>
	 *            if true the system will check the JEM WEB version against the
	 *            JEM NODES version and if there is a mismatch the system will
	 *            exit with error. If false the system will do the chek and in
	 *            case of failure set a warning
	 */
	public void setCheckVersion(Boolean checkVersion) {
		this.checkVersion = checkVersion;
	}

	/**
	 * Returns JEM version
	 * 
	 * @return the jemVersion
	 */
	public String getJemVersion() {
		return jemVersion;
	}

	/**
	 * Sets JEM version
	 * 
	 * @param jemVersion the jemVersion to set
	 */
	public void setJemVersion(String jemVersion) {
		this.jemVersion = jemVersion;
	}

}