/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.server.commons;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.OperatingSystemMXBean;
import java.net.InetSocketAddress;
import java.util.Collection;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import org.pepstock.jem.log.LogAppl;
import org.pepstock.jem.node.ExecutionEnvironment;
import org.pepstock.jem.node.NodeInfo;
import org.pepstock.jem.node.NodeMessage;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.Status;
import org.pepstock.jem.node.configuration.ConfigKeys;

import com.hazelcast.core.IMap;
import com.hazelcast.core.Member;
import com.hazelcast.query.SqlPredicate;

/**
 * A set of methods to manage info of node in the common map for node info
 * 
 * @author Andrea "Stock" Stocchero
 * 
 * @see org.pepstock.jem.node.NodeInfo
 */
public class NodeInfoUtility {

	/**
	 * Factory creates a NodeInfo copying a set of information from Member
	 * object of Hazelcast framework. NodeInfo will use Uuid of Member as the
	 * key.
	 * 
	 * @see org.pepstock.jem.node.NodeInfo
	 * @param member
	 *            member object of Hazelcast framework
	 * @param contextPath Web context path of JEM application, necessary to read manifest for version
	 * @return nodeinfo instance
	 */

	public static final NodeInfo createNodeInfo(Member member,
			String contextPath) {
		NodeInfo info = new NodeInfo();
		// set uuid of member of hazelcast as key
		info.setKey(member.getUuid());
		// set status starting at the beginning
		info.setStatus(Status.STARTING);
		// no supernode (only web member is supernode)
		info.setSuperNode(member.isLiteMember());

		// for net info of member, loads all info inside of nodeinfo
		// port of RMI will be set later
		InetSocketAddress address = member.getInetSocketAddress();
		info.setPort(address.getPort());
		info.setIpaddress(address.getAddress().getHostAddress());
		info.setHostname(address.getHostName());
		// sets label to be displayed by GRS
		info.setLabel(info.getIpaddress() + ":" + info.getPort());
		ExecutionEnvironment executionEnviroment = new ExecutionEnvironment();
		executionEnviroment.setEnvironment(SharedObjects.getInstance()
				.getLocalMember().getConfig().getGroupConfig().getName());
		info.setExecutionEnvironment(executionEnviroment);
		// use JMX to extract current process id, operating system,
		// architecture, processors, system name
		info.setProcessId(ManagementFactory.getRuntimeMXBean().getName());
		OperatingSystemMXBean bean = ManagementFactory
				.getOperatingSystemMXBean();
		info.getNodeInfoBean().setSystemArchitecture(bean.getArch());
		info.getNodeInfoBean().setAvailableProcessors(
				bean.getAvailableProcessors());
		info.getNodeInfoBean().setSystemName(bean.getName());
		
		// reads manifest file for searching version of JEM
		File file = null;
		FileInputStream fis = null;
		try {
			file = new File(contextPath + "META-INF/MANIFEST.MF");
			fis = new FileInputStream(file);
			Manifest manifest = new Manifest(fis);
			// gets JEM vrsion
			Attributes at = manifest.getAttributes(ConfigKeys.JEM_MANIFEST_SECTION);
			String jemVersion = at.getValue(ConfigKeys.JEM_MANIFEST_VERSION);
			// saves JEM version
			if (jemVersion != null) {
				info.setJemVersion(jemVersion);
				SharedObjects.getInstance().setJemVersion(jemVersion);
			}
		} catch (FileNotFoundException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC184W);
		} catch (IOException e) {
			LogAppl.getInstance().emit(NodeMessage.JEMC184W);
		} finally {
			if (fis != null) {
				try {
					fis.close();
				} catch (Exception e) {
				}
			}
		}
		return info;
	}

	/**
	 * Checks if the node was already in the list, using label value (ipaddress
	 * and port). If is in the list, removes old object adding a new one.
	 * 
	 * @param info
	 *            node information object
	 */
	public static synchronized void checkAndStoreNodeInfo(NodeInfo info) {
		// gets map
		IMap<String, NodeInfo> members_map = SharedObjects.getInstance()
				.getLocalMember().getMap(Queues.NODES_MAP);

		// creates a SQL predicate to search the node
		StringBuffer sb = new StringBuffer();
		sb.append("label = '").append(info.getLabel()).append("'");

		try {
			// locks teh key (node key)
			members_map.lock(info.getKey());
			// gets SQL result
			Collection<NodeInfo> nodeInfos = members_map
					.values(new SqlPredicate(sb.toString()));

			// if not empty
			if (!nodeInfos.isEmpty()) {
				// for all nodes. romeve them!
				for (NodeInfo prevNodeInfo : nodeInfos) {
					members_map.remove(prevNodeInfo.getKey());
				}
			}
			// stores the object into map
			members_map.put(info.getKey(), info);
			// boolean checkVersion =
			// Parser.parseBoolean(System.getProperty(ConfigKeys.JEM_CHECK_VERSION),
			// false);
			// check if node release version inside the cluster are different
			Collection<NodeInfo> allNodes = members_map.values();
			for (NodeInfo currNodeInfo : allNodes) {
				if (!currNodeInfo.getJemVersion().equals(info.getJemVersion())) {
					// checks web app has the parameter to stop app if the version is not 
					// the same of JEM group
					// if yes, stop webapp
					if (SharedObjects.getInstance().getCheckVersion()) {
						LogAppl.getInstance().emit(NodeMessage.JEMC191E);
						throw new RuntimeException(NodeMessage.JEMC191E.toMessage().getFormattedMessage());
					} else {
						LogAppl.getInstance().emit(NodeMessage.JEMC185W);
						break;
					}
				}
			}
		} finally {
			// unlocks always the key
			members_map.unlock(info.getKey());
		}
	}

}