/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013  Marco "Fuzzo" Cuccato
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.services;

import java.util.Collection;
import java.util.Map;

import org.pepstock.jem.Job;
import org.pepstock.jem.JobSystemActivity;
import org.pepstock.jem.OutputFileContent;
import org.pepstock.jem.OutputListItem;
import org.pepstock.jem.OutputTree;
import org.pepstock.jem.gwt.client.commons.JobStatus;

import com.google.gwt.user.client.rpc.RemoteService;
import com.google.gwt.user.client.rpc.RemoteServiceRelativePath;

/**
 * The client side stub for the RPC service. The service provides all methods to load the jobs info to display, from different queues and
 * all actions to perform on jobs.
 *  
 */
@RemoteServiceRelativePath(Services.QUEUES)
public interface JobsManagerService extends RemoteService {
	/**
	 * Returns all jobs in input queue, using a job filter name.
	 * 
	 * @param jobNameFilter filter name
	 * @return a collection of jobs in input
	 * @throws Exception if cluster is not available or not authorized
	 */
	Collection<Job> getInputQueue(String jobNameFilter) throws Exception;
	/**
	 * Returns all jobs in running queue, using a job filter name.
	 *  
	 * @param jobNameFilter filter name
	 * @return a collection of jobs in running
	 * @throws Exception if cluster is not available or not authorized
	 */
	Collection<Job> getRunningQueue(String jobNameFilter) throws Exception;
	/**
	 * Returns all jobs in output queue, using a job filter name.
	 *  
	 * @param jobNameFilter filter name
	 * @return a collection of jobs in output
	 * @throws Exception if cluster is not available or not authorized
	 */
	Collection<Job> getOutputQueue(String jobNameFilter) throws Exception;
	/**
	 * Returns all jobs in routing queue, using a job filter name.
	 *  
	 * @param jobNameFilter filter name
	 * @return a collection of jobs in routing
	 * @throws Exception if cluster is not available or not authorized
	 */
	Collection<Job> getRoutingQueue(String jobNameFilter) throws Exception;
	/**
	 * Puts jobs in hold
	 * 
	 * @param jobs list of jobs
	 * @param queueName queue name where the jobs are
	 * @return always true
	 * @throws Exception if cluster is not available or not authorized
	 */
	Boolean hold(Collection<Job> jobs, String queueName) throws Exception;
	/**
	 * Cancels the current execution of job.
	 * 
	 * @param jobs list of jobs
	 * @param force if force cancel is set
	 * @return always true
	 * @throws Exception if cluster is not available or not authorized
	 */
	Boolean cancel(Collection<Job> jobs, boolean force) throws Exception;
	/**
	 * Release jobs previously put in HOLD
	 * 
	 * @param jobs list of jobs
	 * @param queueName queue name where the jobs are
	 * @return always true
	 * @throws Exception if cluster is not available or not authorized
	 */
	Boolean release(Collection<Job> jobs, String queueName) throws Exception;
	
	/**
	 * Submit a new job from inspector of a job
	 * @param content jcl content
	 * @param type JCL type
	 * @return job id
	 * @throws Exception if cluster is not available or not authorized
	 */
	String submit(String content, String type) throws Exception;
	
	/**
	 * Indent JCL during editing
	 * @param content jcl content
	 * @return JCL indented
	 * @throws Exception if cluster is not available or not authorized
	 */
	String indent(String content) throws Exception;
	
	/**
	 * Remove jobs from queue
	 * 
	 * @param jobs list of jobs
	 * @param queueName queue name where the jobs are
	 * @return always true
	 * @throws Exception if cluster is not available or not authorized
	 */
	Boolean purge(Collection<Job> jobs, String queueName) throws Exception;
	
	/**
	 * Some attributes are changed by UI and calls this method to store the new job.
	 * 
	 * @param job job instance related to JCL
	 * @param queueName queue name where the job is
	 * @return always true
	 * @throws Exception if cluster is not available or not authorized
	 */
	Boolean update(Job job, String queueName) throws Exception;
	/**
	 * Returns the JCL content
	 * 
	 * @param job job instance related to JCL
	 * @param queueName queue name where the job is
	 * @return jcl content
	 * @throws Exception if cluster is not available or not authorized
	 */
	String getJcl(Job job, String queueName) throws Exception;
	/**
	 * Returns the tree of the folder where is stored the ouput job
	 * 
	 * @param job job instance
	 * @param queueName queue name where the job is
	 * @return output job content folder  
	 * @throws Exception if cluster is not available or not authorized
	 */
	OutputTree getOutputTree(Job job, String queueName) throws Exception;
	/**
	 * Returns the file content
	 * 
	 * @param job job instance
	 * @param item file representation of output
	 * @return file content
	 * @throws Exception if cluster is not available or not authorized
	 */
	OutputFileContent getOutputFileContent(Job job, OutputListItem item) throws Exception;
	
	/**
	 * Returns a status with a set of collections 
	 * @param filter job name filter or job id
	 * @return JobStatus
	 * @throws Exception if cluster is not available or not authorized 
	 */
	JobStatus getJobStatus(String filter) throws Exception ;
	
	/**
	 * Returns CPU and Memory utilized by job during execution
	 * @param job job instance
	 * @return system activity
	 * @throws Exception if cluster is not available or not authorized 
	 */
	JobSystemActivity getJobSystemActivity(Job job) throws Exception;
	
	/**
	 * Returns the list of JCL type loaded on JEM and their descriptions
	 * 
	 * @return map with all types
	 * @throws Exception if cluster is not available or not authorized
	 */
	Map<String, String> getJclTypes() throws Exception;
	
	
}