/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Simone "Busy" Businaro
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels.swarm;

import org.pepstock.jem.NodeInfoBean;
import org.pepstock.jem.gwt.client.commons.AlertMessageBox;
import org.pepstock.jem.gwt.client.commons.NodeStatusImages;
import org.pepstock.jem.gwt.client.commons.ServiceAsyncCallback;
import org.pepstock.jem.gwt.client.panels.components.AbstractActionsButtonPanel;
import org.pepstock.jem.gwt.client.panels.swarm.commons.EditConfigInspector;
import org.pepstock.jem.gwt.client.panels.swarm.commons.ViewConfigInspector;
import org.pepstock.jem.gwt.client.security.ClientPermissions;
import org.pepstock.jem.gwt.client.services.Services;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.node.configuration.SwarmConfiguration;
import org.pepstock.jem.node.security.Permissions;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.safehtml.shared.SafeHtmlBuilder;
import com.google.gwt.user.client.ui.AbstractImagePrototype;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.HTML;

/**
 * Component with buttons to perform actions on swarm nodes.
 * 
 * @author Simone "Busy" Businaro
 *
 */
public class NodesActions extends AbstractActionsButtonPanel<NodeInfoBean> {
	
	private HTML statusLabel = new HTML();

	/**
	 * 
	 */
    public NodesActions() {
	    super();
	    init();
    }

	@Override
	protected void initButtons() {
		
		addStatus();
		addStartButton();
		addStopButton();
		addConfButton();
//		addEditConfButton();
	}
	
	private void addStatus(){
		add(statusLabel);
		getStatus();
	}
	
	private void setStatus(String statusString){
		SafeHtmlBuilder sb = new SafeHtmlBuilder();
		NodeStatusImages statusObject;
		if (statusString.equals(NodeStatusImages.UNKNOWN.toString())) {
			statusObject = NodeStatusImages.UNKNOWN;
		} else if (statusString.equals(NodeStatusImages.STARTING.toString())) {
			statusObject = NodeStatusImages.STARTING;
		} else if (statusString.equals(NodeStatusImages.INACTIVE.toString())) {
			statusObject = NodeStatusImages.INACTIVE;
		} else if (statusString.equals(NodeStatusImages.ACTIVE.toString())) {
			statusObject = NodeStatusImages.ACTIVE;
		} else if (statusString.equals(NodeStatusImages.DRAINED.toString())) {
			statusObject = NodeStatusImages.DRAINED;
		} else if (statusString.equals(NodeStatusImages.DRAINING.toString())) {
			statusObject = NodeStatusImages.DRAINING;
		} else if (statusString.equals(NodeStatusImages.SHUTTING_DOWN.toString())) {
			statusObject = NodeStatusImages.SHUTTING_DOWN;
		} else {
			// the default!
			statusObject = NodeStatusImages.INACTIVE;
		}
		
		sb.appendHtmlConstant("<table>");
		// adds a label for imgae
		sb.appendHtmlConstant("<tr><td align='left' valign='middle'>Swarm is</td><td width='5px'/><td>");
		// Add the contact image.
		String imageHtml = AbstractImagePrototype.create(statusObject.getImage()).getHTML();
		sb.appendHtmlConstant(imageHtml);
		sb.appendHtmlConstant("</td>");
		// Add the name and address.
		sb.appendHtmlConstant("<td align='left' valign='middle'>");
		sb.appendEscaped(statusString);

		// adds a empty space like a margin 
		sb.appendHtmlConstant("</td><td width='15px'/></tr></table>");
		
		statusLabel.setHTML(sb.toSafeHtml());
	}

	private void addStartButton() {
		// checks if user has the permission to START swarm nodes
		if (ClientPermissions.isAuthorized(Permissions.SWARM, Permissions.SWARM_NODES_START)) {
			Button startButton = new Button("Start", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					start();
				}
			});
			add(startButton);
		}
	}

	private void addStopButton() {
		// checks if user has the permission to STOP swarm nodes
		if (ClientPermissions.isAuthorized(Permissions.SWARM, Permissions.SWARM_NODES_DRAIN)) {
			Button drainButton = new Button("Drain", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					drain();
				}
			});
			add(drainButton);
		}
	}

	private void addConfButton() {
		// checks if user has the permission to EDIT swarm configuration
		if (ClientPermissions.isAuthorized(Permissions.SWARM, Permissions.SWARM_NODES_EDIT_CONFIG)) {
			Button editButton = new Button("Edit Config", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					edit();
				}
			});
			add(editButton);
		}
		else if (ClientPermissions.isAuthorized(Permissions.SWARM, Permissions.SWARM_NODES_VIEW_CONFIG)) {
			Button viewButton = new Button("View Config", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					view();
				}
			});
			add(viewButton);
		}
	}

	/**
	 * Stop all swarm nodes that belong to this environment. A Swarm node is an hazelcast node included in each web node that can either been start or shutdown
	 */
	private void drain() {
		getLoading().startProcessing();
		
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.SWARM_NODES_MANAGER.drain(new ServiceAsyncCallback<Boolean>() {

					@Override
					public void onJemSuccess(Boolean result) {
						if (result){
							AlertMessageBox alert = new AlertMessageBox(Message.INFO, "Swarm stopped!", 
									"Swarm is stopped correctly!");
							alert.open();
						} else {
							AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "Stop command warning!", 
									"Some lite members are not stopped in swarm!");
							alert.open();
						}
						getStatus();
						// if has success, refresh the data, to see in table that they are started
						if (getSearcher() != null){
							getSearcher().refresh();
						}
					}

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Stop command error!", 
								caught.getMessage());
						alert.open();
					}

					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }
				});
			
			}
	    });

	}

	/**
	 * Start all swarm nodes that belong to this environment. A Swarm node is an hazelcast node included in each web node that can either been start or shutdown
	 */
	private void start() {
		getLoading().startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.SWARM_NODES_MANAGER.start(new ServiceAsyncCallback<Boolean>() {

					@Override
					public void onJemSuccess(Boolean result) {
						if (result){
							AlertMessageBox alert = new AlertMessageBox(Message.INFO, "Swarm started!", 
									"Swarm is started correctly!");
							alert.open();
						} else {
							AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "Start command warning!", 
									"Some lite members are not started in swarm!");
							alert.open();
						}
						getStatus();
						// if has success, refresh the data, to see in table that they are started
						if (getSearcher() != null){
							getSearcher().refresh();
						}
					}

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Start command error!", 
								caught.getMessage());
						alert.open();
					}
					
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }

				});
			
			}
	    });

	}
	
	/**
	 * Gets status of swarm changing the value
	 */
	public void getStatus() {
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.SWARM_NODES_MANAGER.getStatus(new ServiceAsyncCallback<String>() {

					@Override
					public void onJemSuccess(String result) {
						if (result != null){
							setStatus(result);
						}
					}

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "getStatus ERROR!", 
								caught.getMessage());
						alert.open();
					}
					
					@Override
                    public void onJemExecuted() {
                    }

				});
			
			}
	    });

	}
	/**
	 * @param nodes collection of nodes to start
	 */
	private void edit() {
		getLoading().startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
    			Services.ROUTING_CONFIG_MANAGER.getSwarmConfiguration(SwarmConfiguration.DEFAULT_NAME, new ServiceAsyncCallback<SwarmConfiguration>() {

    				@Override
    				public void onJemFailure(Throwable caught) {
    					AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Get configuration error!", 
    							caught.getMessage());
    					alert.open();
    				}

    				@Override
    				public void onJemSuccess(SwarmConfiguration result) {
    					if (result != null){
    						//Window.alert(result.toString());
    						try {
    							EditConfigInspector inspector = new EditConfigInspector(result);
    							inspector.setModal(true);
    							inspector.setTitle("Swarm Configuration");
    							inspector.center();
    						} catch (Exception e) {
    							AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "EditConfig error!", 
    									e.getMessage());
    							alert.open();
    						}     							
    					} else {
    						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "Configuration empty!", 
    								"The result of swarm configuration is empty!<br>Please contact your JEM adminitrator.");
    						alert.open();
    					}
    				}
    				
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }

    			});		
			}
	    });

	}
	
	/**
	 * @param nodes collection of nodes to start
	 */
	private void view() {
		getLoading().startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
    			Services.ROUTING_CONFIG_MANAGER.getSwarmConfiguration(SwarmConfiguration.DEFAULT_NAME, new ServiceAsyncCallback<SwarmConfiguration>() {

    				@Override
    				public void onJemFailure(Throwable caught) {
    					AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Get configuration error!", 
    							caught.getMessage());
    					alert.open();
    				}

    				@Override
    				public void onJemSuccess(SwarmConfiguration result) {
    					if (result != null){
    						try {
    							ViewConfigInspector inspector = new ViewConfigInspector(result);
    							inspector.setModal(true);
    							inspector.setTitle("Swarm Configuraton");
    							inspector.center();
    						} catch (Exception e) {
    							AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "ViewConfig error!", 
    									e.getMessage());
    							alert.open();
    						} 
    					} else {
    						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "Configuration empty!", 
    								"The result of swarm configuration is empty!<br>Please contact your JEM adminitrator.");
    						alert.open();
    					}
    				}
    				
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }

    			});		
			}
	    });

	}
}