package org.pepstock.jem.gwt.client.panels.resources.inspector.custom;

import java.util.HashMap;
import java.util.Set;

import org.pepstock.jem.gwt.client.Sizes;
import org.pepstock.jem.gwt.client.commons.CSVUtil;
import org.pepstock.jem.gwt.client.commons.Toast;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.node.resources.ResourceProperty;
import org.pepstock.jem.node.resources.custom.fields.TextListFieldDescriptor;

/**
 * Builds a text-based properties editor, in which the user can free type values.
 * Rendered with a {@link ListEditor}
 * @author Marco "Fuzzo" Cuccato
 */
public class ListFieldPanel extends AbstractFieldPanel<TextListFieldDescriptor, ListEditor, String[]> implements ListEditorListener<String[]> {

	/**
	 * Builds the panel
	 * @param descriptor the descriptor who knows how to render the panel
	 * @param panel the parent panel
	 */
	public ListFieldPanel(TextListFieldDescriptor descriptor, PagePropertiesPanel panel) {
		super(descriptor, panel);
		build();
	}

	@Override
	protected void build() {
		inputObject = new ListEditor(getDescriptor().getInputLabel(), "Values", getDescriptor().getRegExValidator());
		inputObject.setSize(Sizes.HUNDRED_PERCENT, Sizes.HUNDRED_PERCENT);
		
		ResourceProperty existingProperty = getPanel().getResource().getProperties().get(getDescriptor().getKey());
		if (existingProperty == null && getDescriptor().hasDefaultValues()) {
			Set<String> defaultValue = getDescriptor().getDefaultValues();
			String[] defaultValueArray = defaultValue.toArray(new String[0]);
			setSelectedValue(defaultValueArray);
			saveProperty(defaultValueArray);
		}
		
		inputObject.setListener(this);
	}

	@Override
	public boolean checkMandatory() {
		if (getDescriptor().isMandatory()) {
			if (inputObject.getValues().length > 0) {
				return true;
			}
			return false;
		}
		return true;
	}

	@Override
	public boolean validate() {
		boolean allMatch = true;
		String regExValidator = getDescriptor().getRegExValidator();
		String notMatchingValue = null;
		
		if (regExValidator != null && !regExValidator.trim().isEmpty()) {
			try {
				// validator exists, try to validate value
				for (String currentValue : getSelectedValue()) {
					if (currentValue.matches(regExValidator)) {
						allMatch &= true;
					} else {
						notMatchingValue = currentValue;
						allMatch = false;
						break;
					}
				}
			} catch (Exception pse) {
				Toast t = new Toast(Message.ERROR, "Unable to parse regular expression '" + regExValidator + "'. Check the Resource Descriptor.", "Validator error");
				t.show();
				return false;
			}
		}
		
		// allMatch is true if all values are valid or if no validation is needed
		if (allMatch) {
			return true;
		}
		
		Toast t = new Toast(Message.WARNING, "The value '" + notMatchingValue + "' in list must match the regular expression '" + regExValidator + "'.", "Illegal value");
		t.show();
		return false;
	}

	@Override
	public void loadProperties() {
		HashMap<String, ResourceProperty> props = getPanel().getResource().getProperties();
		ResourceProperty p = props.get(getDescriptor().getKey());
		if (p != null && p.getValue() != null) {
			String[] toBeSelectedValues = CSVUtil.split(p.getValue());
			setSelectedValue(toBeSelectedValues);
		}
	}

	@Override
	public String[] getSelectedValue() {
		return inputObject.getValues();
	}

	@Override
	public void setSelectedValue(String[] value) {
		inputObject.setValues(value);
	}

	@Override
	public void valuesChanged(String[] newValues) {
		saveProperty(newValues);
	}

	@Override
	public void saveProperty(String[] value) {
		String key = getDescriptor().getKey();
		String csv = CSVUtil.getCSVPhrase(value);
		getPanel().getResource().setProperty(key, csv);
		setCommonPropertyAttributes();
	}

}
