/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels.nodes;

import java.util.Iterator;
import java.util.LinkedList;

import org.pepstock.jem.Jcl;
import org.pepstock.jem.NodeInfoBean;
import org.pepstock.jem.gwt.client.commons.AbstractTable;
import org.pepstock.jem.gwt.client.commons.AnchorTextColumn;
import org.pepstock.jem.gwt.client.commons.IndexedColumnComparator;
import org.pepstock.jem.gwt.client.commons.InspectListener;
import org.pepstock.jem.gwt.client.commons.NodeStatusImages;
import org.pepstock.jem.gwt.client.commons.TextFilterableHeader;
import org.pepstock.jem.gwt.client.commons.UpdateListener;
import org.pepstock.jem.gwt.client.security.ClientPermissions;
import org.pepstock.jem.node.security.Permissions;
import org.pepstock.jem.util.filters.fields.NodeFilterFields;

import com.google.gwt.cell.client.Cell.Context;
import com.google.gwt.cell.client.CheckboxCell;
import com.google.gwt.cell.client.EditTextCell;
import com.google.gwt.cell.client.FieldUpdater;
import com.google.gwt.cell.client.ValueUpdater;
import com.google.gwt.dom.client.Style.Unit;
import com.google.gwt.safehtml.shared.SafeHtmlBuilder;
import com.google.gwt.user.cellview.client.CellTable;
import com.google.gwt.user.cellview.client.Column;
import com.google.gwt.user.cellview.client.Header;
import com.google.gwt.user.cellview.client.TextColumn;
import com.google.gwt.user.client.ui.AbstractImagePrototype;
import com.google.gwt.view.client.SelectionModel;

/**
 * Creates all columns to show into table, defining the sorter too.
 * 
 * @author Andrea "Stock" Stocchero
 * @version 1.0	
 *
 */
public class NodesTable extends AbstractTable<NodeInfoBean> {

	/**
	 * Adds all columns to table, defining the sort columns too.
	 */
	@Override
	public IndexedColumnComparator<NodeInfoBean> initCellTable(CellTable<NodeInfoBean> table) {
		/*-------------------------+
		 | CHECK BOX FOR SELECTION |
		 +-------------------------*/
		@SuppressWarnings("unchecked")
		final SelectionModel<NodeInfoBean> selectionModel = (SelectionModel<NodeInfoBean>) table.getSelectionModel();
		Column<NodeInfoBean, Boolean> checkColumn = new Column<NodeInfoBean, Boolean>(
				new CheckboxCell(true, false)) {
			@Override
			public Boolean getValue(NodeInfoBean nodeInfoBean) {
				return selectionModel.isSelected(nodeInfoBean);
			}
		};

		CheckboxCell headerCheckBox = new CheckboxCell(true, false);
		Header<Boolean> checkHeader = new Header<Boolean>(headerCheckBox) {
			// imposta lo stato dell'header!
			@Override
			public Boolean getValue() {
				// se e' vuoto, niente e' selezionato/selezionabile
				if (getTable().getVisibleItems().isEmpty()) {
					return false;
				}
				
				// altrimenti testo
				for (NodeInfoBean n : getTable().getVisibleItems()) {
					// se almeno un elemento non e' selezionato, l'header non deve essere selezionato
					if (!getTable().getSelectionModel().isSelected(n)) {
						return false;
					}
				}
				// altrimenti se arrivo qui, tutti gli elementi sono selezionati
				return true;
			}
		};
		
		// updater che seleziona o deseleziona tutti gli elementi visibili in base al "valore" dell'header
		checkHeader.setUpdater(new ValueUpdater<Boolean>() {
			@Override
			public void update(Boolean value) {
				for (NodeInfoBean n : getTable().getVisibleItems()) {
					getTable().getSelectionModel().setSelected(n, value);
				}
			}
		});
				
		table.setColumnWidth(checkColumn, 23, Unit.PX);
		table.addColumn(checkColumn, checkHeader);

		
		/*-------------------------+
		 | IPADDRESS AND PORT      |
		 +-------------------------*/
	    // construct a column that uses anchorRenderer
	    AnchorTextColumn<NodeInfoBean> name = new AnchorTextColumn<NodeInfoBean>() {
			@Override
			public String getValue(NodeInfoBean object) {
				return object.getLabel();
			}

			@Override
			public void onClick(int index, NodeInfoBean object, String value) {
				getInspectListener().inspect(object);
			}
		};
		name.setSortable(true);
		table.addColumn(name, new TextFilterableHeader("Name", NodeFilterFields.NAME.getName()));
		
		/*-------------------------+
		 | HOST NAME               |
		 +-------------------------*/
		TextColumn<NodeInfoBean> hostname = new TextColumn<NodeInfoBean>() {
			@Override
			public String getValue(NodeInfoBean nodeInfoBean) {
				return nodeInfoBean.getHostname();
			}
		};
		hostname.setSortable(true);
		table.addColumn(hostname, new TextFilterableHeader("Hostname", NodeFilterFields.HOSTNAME.getName()));

		/*-------------------------+
		 | DOMAIN                  |
		 +-------------------------*/
		
		if (ClientPermissions.isAuthorized(Permissions.NODES, Permissions.NODES_UPDATE)){
			Column<NodeInfoBean, String> domain = new Column<NodeInfoBean, String>(
					new EditTextCell()) {
				@Override
				public String getValue(NodeInfoBean nodeInfoBean) {
					return nodeInfoBean.getExecutionEnvironment().getDomain();
				}
			};
			domain.setSortable(true);
			domain.setFieldUpdater(new FieldUpdater<NodeInfoBean, String>() {
				@Override
				public void update(int index, NodeInfoBean nodeInfoBean, String value) {
					if (value !=null){
						if (value.trim().length() == 0){
							value = Jcl.DEFAULT_DOMAIN;
						}
						if (!value.equalsIgnoreCase(nodeInfoBean.getExecutionEnvironment().getDomain())){
							nodeInfoBean.getExecutionEnvironment().setDomain(value);
							updateNode(nodeInfoBean);
						}
						return;
					}
					refresh();
				}
			});	
			table.addColumn(domain, new TextFilterableHeader("Domain", NodeFilterFields.DOMAIN.getName()));
		} else {
			TextColumn<NodeInfoBean> domain = new TextColumn<NodeInfoBean>() {
				@Override
				public String getValue(NodeInfoBean nodeInfoBean) {
					return nodeInfoBean.getExecutionEnvironment().getDomain();
				}
			};
			domain.setSortable(true);
			table.addColumn(domain, new TextFilterableHeader("Domain", NodeFilterFields.DOMAIN.getName()));
		}
		
		/*-------------------------+
		 | STATIC AFFINITIES       |
		 +-------------------------*/
		if (ClientPermissions.isAuthorized(Permissions.NODES, Permissions.NODES_UPDATE)){
			Column<NodeInfoBean, String> affinity = new Column<NodeInfoBean, String>(
					new EditTextCell()) {
				@Override
				public String getValue(NodeInfoBean nodeInfoBean) {
					return toAffinityString(nodeInfoBean.getExecutionEnvironment().getStaticAffinities());
				}
			};
			affinity.setSortable(true);
			affinity.setFieldUpdater(new FieldUpdater<NodeInfoBean, String>() {
				@Override
				public void update(int index, NodeInfoBean nodeInfoBean, String value) {
					if (value !=null){
						if (value.trim().length() == 0){
							value = Jcl.DEFAULT_AFFINITY;
						}
						nodeInfoBean.getExecutionEnvironment().getStaticAffinities().clear();
						String[] affinities = value.split(",");
						for (int i=0; i<affinities.length; i++){
							nodeInfoBean.getExecutionEnvironment().getStaticAffinities().add(affinities[i]);
						}
						updateNode(nodeInfoBean);
						return;
					}
					refresh();
				}
			});	
			table.addColumn(affinity, new TextFilterableHeader("Static Affinities", NodeFilterFields.STATIC_AFFINITIES.getName()));
		} else {
			TextColumn<NodeInfoBean> affinity = new TextColumn<NodeInfoBean>() {
				@Override
				public String getValue(NodeInfoBean nodeInfoBean) {
					return toAffinityString(nodeInfoBean.getExecutionEnvironment().getStaticAffinities());
				}
			};
			affinity.setSortable(true);
			table.addColumn(affinity, new TextFilterableHeader("Static Affinities", NodeFilterFields.STATIC_AFFINITIES.getName()));
		}
		
		/*-------------------------+
		 | DYNAMIC AFFINITIES      |
		 +-------------------------*/
	    TextColumn<NodeInfoBean> dynAffinity = new TextColumn<NodeInfoBean>() {
	    	@Override
	    	public String getValue(NodeInfoBean nodeInfoBean) {
	    		return toAffinityString(nodeInfoBean.getExecutionEnvironment().getDynamicAffinities());
	    	}
	    };
	    dynAffinity.setSortable(true);
	    table.addColumn(dynAffinity, new TextFilterableHeader("Dynamic Affinities", NodeFilterFields.DYNAMIC_AFFINITIES.getName()));
		
		/*-------------------------+
		 | STATUS                  |
		 +-------------------------*/
		TextColumn<NodeInfoBean> status = new TextColumn<NodeInfoBean>() {

			@Override
			public String getValue(NodeInfoBean object) {
				if (object == null || object.getStatus() == null) {
					return "";
				}
				return object.getStatus();	
			}

			@Override
			public void render(Context context, NodeInfoBean object, SafeHtmlBuilder sb) {
				if (object == null || object.getStatus() == null || object.getStatus().trim().isEmpty()) {
					return;
				}
				String statusString = object.getStatus();
				NodeStatusImages statusObject;
				if (statusString.equals(NodeStatusImages.UNKNOWN.toString())) {
					statusObject = NodeStatusImages.UNKNOWN;
				} else if (statusString.equals(NodeStatusImages.STARTING.toString())) {
					statusObject = NodeStatusImages.STARTING;
				} else if (statusString.equals(NodeStatusImages.INACTIVE.toString())) {
					statusObject = NodeStatusImages.INACTIVE;
				} else if (statusString.equals(NodeStatusImages.ACTIVE.toString())) {
					statusObject = NodeStatusImages.ACTIVE;
				} else if (statusString.equals(NodeStatusImages.DRAINED.toString())) {
					statusObject = NodeStatusImages.DRAINED;
				} else if (statusString.equals(NodeStatusImages.DRAINING.toString())) {
					statusObject = NodeStatusImages.DRAINING;
				} else if (statusString.equals(NodeStatusImages.SHUTTING_DOWN.toString())) {
					statusObject = NodeStatusImages.SHUTTING_DOWN;
				} else {
					// the default!
					statusObject = NodeStatusImages.INACTIVE;
				}
				
				sb.appendHtmlConstant("<table>");
				// Add the contact image.
				sb.appendHtmlConstant("<tr><td>");
				String imageHtml = AbstractImagePrototype.create(statusObject.getImage()).getHTML();
				sb.appendHtmlConstant(imageHtml);
				sb.appendHtmlConstant("</td>");
				// Add the name and address.
				sb.appendHtmlConstant("<td align='left' valign='middle'>");
				sb.appendEscaped(statusString);
				if (!object.isOperational())
					sb.appendEscaped(" (not operational)");	
				sb.appendHtmlConstant("</td></tr></table>");
			}
		};
		status.setSortable(true);
		table.addColumn(status, new TextFilterableHeader("Status", NodeFilterFields.STATUS.getName()));

		
		/*-------------------------+
		 | OS NAME                 |
		 +-------------------------*/
		TextColumn<NodeInfoBean> systemName = new TextColumn<NodeInfoBean>() {
			@Override
			public String getValue(NodeInfoBean nodeInfoBean) {
				return nodeInfoBean.getSystemName();
			}
		};
		systemName.setSortable(true);
		table.addColumn(systemName, new TextFilterableHeader("OS", NodeFilterFields.OS.getName()));

		/*-------------------------+
		 | CURRENT JOB             |
		 +-------------------------*/
		TextColumn<NodeInfoBean> currentJob = new TextColumn<NodeInfoBean>() {
			@Override
			public String getValue(NodeInfoBean nodeInfoBean) {
				return (nodeInfoBean.getJobName() == null) ?  "" : nodeInfoBean.getJobName();
			}
		};
		currentJob.setSortable(true);
		table.addColumn(currentJob, new TextFilterableHeader("Current Job", NodeFilterFields.CURRENT_JOB.getName()));

		
		return new NodesComparator(1);

	}


	private String toAffinityString(LinkedList<String> affinities) {
		if (affinities.isEmpty())
			return "";
		Iterator<String> i = affinities.iterator();
		StringBuilder sb = new StringBuilder();
		for (;;) {
			String aff = i.next();
			sb.append(aff);
			if (! i.hasNext())
				return sb.toString();
			sb.append(",");
		}
	}

	
	private void refresh(){
		updateNode(null);
	}
	
	private void updateNode(NodeInfoBean node){
		InspectListener<NodeInfoBean> listener = getInspectListener();
		if (listener instanceof UpdateListener<?>){
			UpdateListener<NodeInfoBean> ulistener= (UpdateListener<NodeInfoBean>) listener;
			ulistener.update(node);
		}
	}
}