/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels.nodes;

import java.util.Collection;

import org.pepstock.jem.NodeInfoBean;
import org.pepstock.jem.gwt.client.commons.AlertMessageBox;
import org.pepstock.jem.gwt.client.commons.ServiceAsyncCallback;
import org.pepstock.jem.gwt.client.panels.components.AbstractActionsButtonPanel;
import org.pepstock.jem.gwt.client.security.ClientPermissions;
import org.pepstock.jem.gwt.client.services.Services;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.node.security.Permissions;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.view.client.MultiSelectionModel;

/**
 * Component with buttons to perform actions on selected nodes.
 * 
 * @author Andrea "Stock" Stocchero
 *
 */
public class NodesActions extends AbstractActionsButtonPanel<NodeInfoBean> {

	/**
	 * 
	 */
    public NodesActions() {
	    super();
	    init();
    }

	@Override
	protected void initButtons() {
		addStartButton();
		addDrainButton();
	}

	private void addStartButton() {
		// checks if user has the permission to START job
		if (ClientPermissions.isAuthorized(Permissions.NODES, Permissions.NODES_START)) {
			Button startButton = new Button("Start", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					@SuppressWarnings("unchecked")
					MultiSelectionModel<NodeInfoBean> selectionModel = (MultiSelectionModel<NodeInfoBean>) getUnderlyingTable().getTable().getSelectionModel();
					if (selectionModel.getSelectedSet().isEmpty()) {
						// do nothing
						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "No node selected!", 
								"No node is selected and it's not possible to perform START command.");
						alert.open();
						return;
					}
					// do!
					start(selectionModel.getSelectedSet());
					selectionModel.clear();
				}
			});
			add(startButton);
		}
	}

	private void addDrainButton() {
		// checks if user has the permission to DRAIN job
		if (ClientPermissions.isAuthorized(Permissions.NODES, Permissions.NODES_DRAIN)) {
			Button drainButton = new Button("Drain", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					// gets the selected jobs
					@SuppressWarnings("unchecked")
					MultiSelectionModel<NodeInfoBean> selectionModel = (MultiSelectionModel<NodeInfoBean>) getUnderlyingTable().getTable().getSelectionModel();
					if (selectionModel.getSelectedSet().isEmpty()) {
						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "No node selected!", 
								"No node is selected and it's not possible to perform DRAIN command.");
						alert.open();
						return;
					}
					// do!
					drain(selectionModel.getSelectedSet());
					// clear selection
					selectionModel.clear();
				}
			});
			add(drainButton);
		}
	}


	/**
	 * @param nodes collection of nodes to drain
	 */
	private void drain(final Collection<NodeInfoBean> nodes) {
		getLoading().startProcessing();
		
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.NODES_MANAGER.drain(nodes, new ServiceAsyncCallback<Boolean>() {

					@Override
					public void onJemSuccess(Boolean result) {
						// if has success, refresh the data, to see in table that they are drained
						if (getSearcher() != null)
							getSearcher().refresh();
					}

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Drain command error!", 
								caught.getMessage());
						alert.open();

					}
					
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }
				});
			
			}
	    });

	}

	/**
	 * @param nodes collection of nodes to start
	 */
	private void start(final Collection<NodeInfoBean> nodes) {
		getLoading().startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.NODES_MANAGER.start(nodes, new ServiceAsyncCallback<Boolean>() {

					@Override
					public void onJemSuccess(Boolean result) {
						// if has success, refresh the data, to see in table that they are started
						if (getSearcher() != null)
							getSearcher().refresh();
					}

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Start command error!", 
								caught.getMessage());
						alert.open();
					}
					
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }

				});
			
			}
	    });

	}

}