/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels.jobs.commons;

import java.util.Collection;

import org.pepstock.jem.Job;
import org.pepstock.jem.gwt.client.commons.AlertMessageBox;
import org.pepstock.jem.gwt.client.commons.ConfirmMessageBox;
import org.pepstock.jem.gwt.client.commons.HideHandler;
import org.pepstock.jem.gwt.client.commons.PreferredButton;
import org.pepstock.jem.gwt.client.commons.ServiceAsyncCallback;
import org.pepstock.jem.gwt.client.security.ClientPermissions;
import org.pepstock.jem.gwt.client.services.Services;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.node.Queues;
import org.pepstock.jem.node.security.Permissions;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.PopupPanel;
import com.google.gwt.view.client.MultiSelectionModel;

/**
 * Component with buttons to perform actions on selected jobs.
 * @author Andrea "Stock" Stocchero
 */
public class JobsBaseActions extends AbstractJobsActions {

	private String queueName = null;

	/**
	 * Constructs all UI, adding buttons
	 * @param queueName jobs queue name to show 
	 */
	public JobsBaseActions(String queueName) {
		this.queueName = queueName;
		init();
	}
	
	@Override
	protected void initButtons() {
		addPurgeButton();
		addHoldButton();
		addReleaseButton();
		if (queueName != null){
			if (queueName.equalsIgnoreCase(Queues.INPUT_QUEUE))
				addSubmitButton();
		}
	}

	private void addReleaseButton() {
		// checks if user has the permission to RELEASE job 
		if (ClientPermissions.isAuthorized(Permissions.JOBS, Permissions.JOBS_RELEASE)){
			Button releaseButton = new Button("Release", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					// gets the selected jobs
					@SuppressWarnings("unchecked")
					MultiSelectionModel<Job> selectionModel = (MultiSelectionModel<Job>) getUnderlyingTable().getTable().getSelectionModel();
					if (selectionModel.getSelectedSet().isEmpty()) {
						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "No job selected!", 
								"No job is selected and it's not possible to perform RELEASE command.");
						alert.open();
						return;
					}
					// do!
					release(selectionModel.getSelectedSet());
					// clear selection
					selectionModel.clear();
				}
			});
			add(releaseButton);
		}
	}
	
	private void addHoldButton() {
		// checks if user has the permission to HOLD job 
		if (ClientPermissions.isAuthorized(Permissions.JOBS, Permissions.JOBS_HOLD)){
			Button holdButton = new Button("Hold", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					// gets the selected jobs
					@SuppressWarnings("unchecked")
					MultiSelectionModel<Job> selectionModel = (MultiSelectionModel<Job>) getUnderlyingTable().getTable().getSelectionModel();
					if (selectionModel.getSelectedSet().isEmpty()) {
						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "No job selected!", 
								"No job is selected and it's not possible to perform HOLD command.");
						alert.open();
						return;
					}
					// do!
					hold(selectionModel.getSelectedSet());
					// clear selection
					selectionModel.clear();
				}
			});
			add(holdButton);
		}
	}
	
	private void addPurgeButton() {
		// checks if user has the permission to PURGE job 
		if (ClientPermissions.isAuthorized(Permissions.JOBS, Permissions.JOBS_PURGE)){
			Button purgeButton = new Button("Purge", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					// gets the selected jobs
					@SuppressWarnings("unchecked")
					final MultiSelectionModel<Job> selectionModel = (MultiSelectionModel<Job>) getUnderlyingTable().getTable().getSelectionModel();
					if (selectionModel.getSelectedSet().isEmpty()) {
						AlertMessageBox alert = new AlertMessageBox(Message.WARNING, "No job selected!", 
								"No job is selected and it's not possible to perform PURGE command.");
						alert.open();
						return;
					}
					
					ConfirmMessageBox cd = new ConfirmMessageBox("Confirm PURGE", "Are you sure you want to purge the selected jobs?");
			        cd.setHideHandler(new HideHandler() {
						@Override
						public void onHide(PreferredButton button) {
					        if (button.getAction() == PreferredButton.YES_ACTION){
								// do!
								purge(selectionModel.getSelectedSet());
								// clear selection
								selectionModel.clear();
					        }
						}
					});
					cd.open();
				}
			});
			add(purgeButton);
		}
	}
	
	private void addSubmitButton() {
		if (ClientPermissions.isAuthorized(Permissions.JOBS, Permissions.JOBS_SUBMIT)){
			// ALL users can submit jobs 
			Button submitButton = new Button("Submit", new ClickHandler() {
				@Override
				public void onClick(ClickEvent event) {
					// do!
					// creates the inspector and shows it
					Submitter inspector = new Submitter();
					// be carefully about the HEIGHT and WIDTH calculation
					inspector.setWidth("600px");
					inspector.setHeight("240px");
					inspector.setModal(true);
					inspector.setTitle("Submit");
					inspector.center();

					// adds for closing
					inspector.addCloseHandler(new CloseHandler<PopupPanel>() {

						@Override
						public void onClose(CloseEvent<PopupPanel> arg0) {
							if (getSearcher() != null)
								getSearcher().refresh();
						}
					});
				}
			});
			add(submitButton);
		}
	}


	/**
	 * @param jobs collection of jobs to hold
	 */
	private void hold(final Collection<Job> jobs){
		getLoading().startProcessing();
		
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.QUEUES_MANAGER.hold(jobs, queueName, new ServiceAsyncCallback<Boolean>() {
					
					@Override
					public void onJemSuccess(Boolean result) {
						// if has success, refresh the data, to see in table that they are in hold
						if (getSearcher() != null)
							getSearcher().refresh();
					}
					
					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Hold command error!", 
								caught.getMessage());
						alert.open();
					}

					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }
				});
			
			}
	    });

		
	}
	
	/**
	 * @param jobs collections of jobs to release (if they are in hold)
	 */
	private void release(final Collection<Job> jobs){
		getLoading().startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.QUEUES_MANAGER.release(jobs, queueName, new ServiceAsyncCallback<Boolean>() {
					
					@Override
					public void onJemSuccess(Boolean result) {
						// if has success, refresh the data, to see in table that they are not in hold
						if (getSearcher() != null)
							getSearcher().refresh();
					}
					
					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Release command error!", 
								caught.getMessage());
						alert.open();
					}
					
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }

				});
			
			}
	    });

		
	}
	
	/**
	 * @param jobs collections of jobs to purge
	 */
	private void purge(final Collection<Job> jobs){
		getLoading().startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.QUEUES_MANAGER.purge(jobs, queueName, new ServiceAsyncCallback<Boolean>() {
					
					@Override
					public void onJemSuccess(Boolean result) {
						// if has success, refresh the data, to see in table without the removed jobs
						if (getSearcher() != null)
							getSearcher().refresh();
					}
					
					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Purge command error!", 
								caught.getMessage());
						alert.open();
					}
					
					@Override
                    public void onJemExecuted() {
						getLoading().stopProcessing();
                    }

				});
			
			}
	    });

	}

}