/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels.administration.workload;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.IsWidget;
import com.google.gwt.user.client.ui.Widget;
import com.sencha.gxt.chart.client.chart.Chart;
import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.Legend;
import com.sencha.gxt.chart.client.chart.axis.CategoryAxis;
import com.sencha.gxt.chart.client.chart.axis.NumericAxis;
import com.sencha.gxt.chart.client.chart.series.LineSeries;
import com.sencha.gxt.chart.client.chart.series.Primitives;
import com.sencha.gxt.chart.client.chart.series.SeriesLabelProvider;
import com.sencha.gxt.chart.client.chart.series.SeriesToolTipConfig;
import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.RGB;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.TextSprite;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.data.shared.LabelProvider;
import com.sencha.gxt.data.shared.ListStore;
import com.sencha.gxt.widget.core.client.container.SimpleContainer;
import com.sencha.gxt.widget.core.client.container.VerticalLayoutContainer;
import com.sencha.gxt.widget.core.client.container.VerticalLayoutContainer.VerticalLayoutData;

/**
 * @author Andrea "Stock" Stocchero
 * 
 */
public class TimeSeriesChart implements IsWidget {

	private static final DataPropertyAccess dataAccess = GWT.create(DataPropertyAccess.class);

	private static final String TIME_AXIS = "Time";

	/**
	 * 
	 */
	public static final int JOBS_SUBMITTED = 0;

	/**
	 * 
	 */
	public static final int JCLS_CHECKED = 1;

	private final Chart<Data> chart = new Chart<Data>();

	private final ListStore<Data> store = new ListStore<Data>(dataAccess.nameKey());
	
	private NumericAxis<Data> axis = null;
	
	private int width = 0;
	
	private int height = 0;
	
	private int type = JOBS_SUBMITTED;
	
	private boolean loaded = false;
	
	private SimpleContainer panel = new SimpleContainer();

	/**
	 * 
	 * @param type
	 */
	public TimeSeriesChart(int type) {
		setType(type);
		chart.setShadowChart(true);
		chart.setAnimated(true);
	}

	/**
	 * @return the type
	 */
	public int getType() {
		return type;
	}

	/**
	 * @param type
	 *            the type to set
	 */
	public void setType(int type) {
		this.type = type;
	}

	/**
	 * @return the width
	 */
	public int getWidth() {
		return width;
	}

	/**
	 * @param width the width to set
	 */
	public void setWidth(int width) {
		this.width = width;
	}

	/**
	 * @return the height
	 */
	public int getHeight() {
		return height;
	}

	/**
	 * @param height the height to set
	 */
	public void setHeight(int height) {
		this.height = height;
	}

	/**
	 * @return the loaded
	 */
	public boolean isLoaded() {
		return loaded;
	}

	/**
	 * @param loaded the loaded to set
	 */
	public void setLoaded(boolean loaded) {
		this.loaded = loaded;
	}

	/**
	 * 
	 * @param data
	 */
	public void setData(List<Data> data) {
		boolean redraw = (store.size() == 0);
		if (redraw)
			store.addAll(data);
		else 
			store.replaceAll(data);
		chart.setStore(store);
		if (redraw) {
			createNumericAxis();
			createCategoryAxis();
			createLineSeries();
			createLegend();
		}
		axis.setMaximum(max());
		chart.redrawChart();
		setLoaded(true);
	}

	
	private int max(){
		Data data = Collections.max(store.getAll(), new Comparator<Data>() {
			@Override
            public int compare(Data arg0, Data arg1) {
				int diff = 0;
	            if (getType() == JOBS_SUBMITTED){
	            	diff = arg0.getJobsSubmitted() - arg1.getJobsSubmitted();
	            } else {
	            	diff = arg0.getJclsChecked() - arg1.getJclsChecked();
	            }
	            return diff;
            }
		});
		int max = (getType() == JOBS_SUBMITTED) ? data.getJobsSubmitted() : data.getJclsChecked();
		
		max = ((int)(max/10) + 1) * 10;
		return max;
	}
	/**
	 * 
	 */
	public Widget asWidget() {

		VerticalLayoutContainer layout = new VerticalLayoutContainer();
		panel.add(layout);

		chart.setLayoutData(new VerticalLayoutData(1, 1));
		layout.add(chart);
		redraw();
		return panel;
	}
	
	/**
	 * Redraw the chart
	 */
	public void redraw(){
		panel.setPixelSize(width, height);
		chart.redrawChart();
	}
	
	/**
	 * 
	 */
	private void createNumericAxis() {
		axis = new NumericAxis<Data>();
		axis.setPosition(Position.LEFT);
		axis.addField((getType() == JOBS_SUBMITTED) ? dataAccess.jobsSubmitted() : dataAccess.jclsChecked());
		

		TextSprite titleSprite = new TextSprite("Number");
		titleSprite.setFontSize(12);
		axis.setTitleConfig(titleSprite);

		axis.setMinorTickSteps(1);
		axis.setDisplayGrid(true);
		PathSprite odd = new PathSprite();
		odd.setOpacity(1);
		odd.setFill(new Color("#ddd"));
		odd.setStroke(new Color("#bbb"));
		odd.setStrokeWidth(0.5);
		axis.setGridOddConfig(odd);
		axis.setMinimum(0);

		chart.addAxis(axis);
	}

	/**
	 * 
	 */
	private void createCategoryAxis() {
		CategoryAxis<Data, String> catAxis = new CategoryAxis<Data, String>();
		catAxis.setPosition(Position.BOTTOM);
		catAxis.setField(dataAccess.key());

		TextSprite title = new TextSprite(TIME_AXIS);
		title.setFontSize(12);
		catAxis.setTitleConfig(title);

		catAxis.setLabelProvider(new LabelProvider<String>() {
			@Override
			public String getLabel(String item) {
				return item.substring(0, 5);
			}
		});

		chart.addAxis(catAxis);
	}

	/**
	 * 
	 */
	private void createLineSeries() {
		final LineSeries<Data> series = new LineSeries<Data>();
		series.setYAxisPosition(Position.LEFT);
		series.setYField((getType() == JOBS_SUBMITTED) ? dataAccess.jobsSubmitted() : dataAccess.jclsChecked());
		series.setStroke(new RGB(32, 68, 186));
		series.setShowMarkers(true);
		series.setSmooth(true);
		series.setFill(new RGB(32, 68, 186));

		Sprite marker = Primitives.diamond(0, 0, 6);
		marker.setFill(new RGB(32, 68, 186));
		series.setMarkerConfig(marker);

		series.setHighlighting(true);

	    final SeriesToolTipConfig<Data> config = new SeriesToolTipConfig<Data>();
	    config.setLabelProvider(new SeriesLabelProvider<Data>() {
			
			@Override
			public String getLabel(Data item, ValueProvider<? super Data, ? extends Number> valueProvider) {
				return "Count: "+String.valueOf((getType() == JOBS_SUBMITTED) ? item.getJobsSubmitted() : item.getJclsChecked());
			}
		});
	    series.setToolTipConfig(config);
	
		chart.addSeries(series);
	}

	/**
	 * 
	 */
	private void createLegend() {
		final Legend<Data> legend = new Legend<Data>();
		legend.setPosition(Position.RIGHT);
//		legend.setItemHighlighting(true);
//		legend.setItemHiding(true);
		chart.setLegend(legend);

	}
}