/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels.administration.queues.inspector;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.ui.IsWidget;
import com.google.gwt.user.client.ui.Widget;
import com.sencha.gxt.chart.client.chart.Chart;
import com.sencha.gxt.chart.client.chart.Chart.Position;
import com.sencha.gxt.chart.client.chart.Legend;
import com.sencha.gxt.chart.client.chart.axis.CategoryAxis;
import com.sencha.gxt.chart.client.chart.axis.NumericAxis;
import com.sencha.gxt.chart.client.chart.series.LineSeries;
import com.sencha.gxt.chart.client.chart.series.Primitives;
import com.sencha.gxt.chart.client.chart.series.SeriesLabelProvider;
import com.sencha.gxt.chart.client.chart.series.SeriesToolTipConfig;
import com.sencha.gxt.chart.client.draw.Color;
import com.sencha.gxt.chart.client.draw.RGB;
import com.sencha.gxt.chart.client.draw.path.PathSprite;
import com.sencha.gxt.chart.client.draw.sprite.Sprite;
import com.sencha.gxt.chart.client.draw.sprite.TextSprite;
import com.sencha.gxt.core.client.ValueProvider;
import com.sencha.gxt.data.shared.LabelProvider;
import com.sencha.gxt.data.shared.ListStore;
import com.sencha.gxt.widget.core.client.container.SimpleContainer;
import com.sencha.gxt.widget.core.client.container.VerticalLayoutContainer;
import com.sencha.gxt.widget.core.client.container.VerticalLayoutContainer.VerticalLayoutData;

/**
 * @author Andrea "Stock" Stocchero
 * 
 */
public class TimeSeriesChart implements IsWidget {

	private static final DataPropertyAccess dataAccess = GWT.create(DataPropertyAccess.class);

	private static final String TIME_AXIS = "Time";

	@SuppressWarnings("javadoc")
    public static final int ENTRIES = 0, HITS = 1, LOCKED = 2, WAITS = 3, GETS = 4, PUTS = 5, REMOVES = 6;

	private final Chart<Data> chart = new Chart<Data>();

	private final ListStore<Data> store = new ListStore<Data>(dataAccess.nameKey());
	
	private NumericAxis<Data> axis = null;
	
	private int width = 0;
	
	private int height = 0;

	private int type = ENTRIES;
	
	private boolean loaded = false;

	/**
	 * 
	 * @param type
	 */
	public TimeSeriesChart(int type) {
		setType(type);
		chart.setShadowChart(true);
		chart.setAnimated(true);
	}

	/**
	 * @return the type
	 */
	public int getType() {
		return type;
	}

	/**
	 * @param type
	 *            the type to set
	 */
	public void setType(int type) {
		this.type = type;
	}

	/**
	 * @return the width
	 */
	public int getWidth() {
		return width;
	}

	/**
	 * @param width the width to set
	 */
	public void setWidth(int width) {
		this.width = width;
	}

	/**
	 * @return the height
	 */
	public int getHeight() {
		return height;
	}

	/**
	 * @param height the height to set
	 */
	public void setHeight(int height) {
		this.height = height;
	}

	/**
	 * @return the loaded
	 */
	public boolean isLoaded() {
		return loaded;
	}

	/**
	 * @param loaded the loaded to set
	 */
	public void setLoaded(boolean loaded) {
		this.loaded = loaded;
	}

	/**
	 * 
	 * @param data
	 */
	public void setData(List<Data> data) {
		boolean redraw = (store.size() == 0);
		if (redraw)
			store.addAll(data);
		else 
			store.replaceAll(data);
		chart.setStore(store);
		if (redraw) {
			createNumericAxis();
			createCategoryAxis();
			createLineSeries();
			createLegend();
		}
		axis.setMaximum(max());
		chart.redrawChart();
		setLoaded(true);
	}

	
	private int max(){
		Data data = Collections.max(store.getAll(), new Comparator<Data>() {
			@Override
            public int compare(Data arg0, Data arg1) {
				long diff = 0;
				switch (getType()){
				case ENTRIES: diff = arg0.getEntries() - arg1.getEntries();
				break;
				case HITS: diff = arg0.getHits() - arg1.getHits();
				break;
				case LOCKED: diff = arg0.getLocked() - arg1.getLocked();
				break;
				case WAITS: diff = arg0.getLockWaits() - arg1.getLockWaits();
				break;
				case GETS: diff = arg0.getGets()- arg1.getGets();
				break;
				case PUTS: diff = arg0.getPuts() - arg1.getPuts();
				break;
				case REMOVES: diff = arg0.getRemoves() - arg1.getRemoves();
				break;
										
				default: diff = arg0.getEntries() - arg1.getEntries();						
										
				}
	            return (int)diff;
            }
		});
		long max = 0;
		switch (getType()){
		case ENTRIES: max = data.getEntries();
		break;
		case HITS: max = data.getHits();
		break;
		case LOCKED: max = data.getLocked();
		break;
		case WAITS: max = data.getLockWaits();
		break;
		case GETS: max = data.getGets();
		break;
		case PUTS: max = data.getPuts();
		break;
		case REMOVES: max = data.getRemoves();
		break;
								
		default: max = data.getEntries();						
								
		}
		
		max = ((long)(max/10) + 1) * 10;
		return (int)max;
	}
	/**
	 * 
	 */
	public Widget asWidget() {
		
		SimpleContainer panel = new SimpleContainer();
		panel.setPixelSize(width, height);

		VerticalLayoutContainer layout = new VerticalLayoutContainer();
		panel.add(layout);

		chart.setLayoutData(new VerticalLayoutData(1, 1));
		layout.add(chart);
		return panel;
	}

	/**
	 * 
	 */
	private void createNumericAxis() {
		axis = new NumericAxis<Data>();
		axis.setPosition(Position.LEFT);
		
		switch (getType()){
		case ENTRIES: axis.addField(dataAccess.entries());
		break;
		case HITS: axis.addField(dataAccess.hits());
		break;
		case LOCKED: axis.addField(dataAccess.locked());
		break;
		case WAITS: axis.addField(dataAccess.lockWaits());
		break;
		case GETS: axis.addField(dataAccess.gets());
		break;
		case PUTS: axis.addField(dataAccess.puts());
		break;
		case REMOVES: axis.addField(dataAccess.removes());
		break;
								
		default: axis.addField(dataAccess.entries());						
								
		}

		TextSprite titleSprite = null;
		
		if ((getType() == ENTRIES) || (getType() == LOCKED)){
			titleSprite = new TextSprite("Entries");
		} else {
			titleSprite = new TextSprite("Count");
		}
		titleSprite.setFontSize(12);
		axis.setTitleConfig(titleSprite);
		
		axis.setMinorTickSteps(1);
		axis.setDisplayGrid(true);
		PathSprite odd = new PathSprite();
		odd.setOpacity(1);
		odd.setFill(new Color("#ddd"));
		odd.setStroke(new Color("#bbb"));
		odd.setStrokeWidth(0.5);
		axis.setGridOddConfig(odd);
		axis.setMinimum(0);

		chart.addAxis(axis);
	}

	/**
	 * 
	 */
	private void createCategoryAxis() {
		CategoryAxis<Data, String> catAxis = new CategoryAxis<Data, String>();
		catAxis.setPosition(Position.BOTTOM);
		catAxis.setField(dataAccess.key());

		TextSprite title = new TextSprite(TIME_AXIS);
		title.setFontSize(12);
		catAxis.setTitleConfig(title);

		catAxis.setLabelProvider(new LabelProvider<String>() {
			@Override
			public String getLabel(String item) {
				return item.substring(0, 5);
			}
		});

		chart.addAxis(catAxis);
	}

	/**
	 * 
	 */
	private void createLineSeries() {
	
		final LineSeries<Data> series = new LineSeries<Data>();
		series.setYAxisPosition(Position.LEFT);
		
		switch (getType()){
		case ENTRIES: series.setYField(dataAccess.entries());
		break;
		case HITS: series.setYField(dataAccess.hits());
		break;
		case LOCKED: series.setYField(dataAccess.locked());
		break;
		case WAITS: series.setYField(dataAccess.lockWaits());
		break;
		case GETS: series.setYField(dataAccess.gets());
		break;
		case PUTS: series.setYField(dataAccess.puts());
		break;
		case REMOVES: series.setYField(dataAccess.removes());
		break;
								
		default: series.setYField(dataAccess.entries());						
								
		}

		series.setStroke(new RGB(148,174,10));
		series.setShowMarkers(true);
		series.setSmooth(true);
		series.setFill(new RGB(148,174,10));

		Sprite marker = Primitives.triangle(0, 0, 6);
		marker.setFill(new RGB(148,174,10));
		series.setMarkerConfig(marker);

		series.setHighlighting(true);

		chart.addSeries(series);
		
	    final SeriesToolTipConfig<Data> config = new SeriesToolTipConfig<Data>();
	    config.setLabelProvider(new SeriesLabelProvider<Data>() {
			
			@Override
			public String getLabel(Data item, ValueProvider<? super Data, ? extends Number> valueProvider) {
				String message = null;
				
				switch (getType()){
				case ENTRIES: message = "Entries: "+item.getEntries();
				break;
				case HITS: message = "Count of hits: "+item.getHits();
				break;
				case LOCKED: message = "Count of locks: "+item.getLocked();
				break;
				case WAITS: message = "Count of waiters: "+item.getLockWaits();
				break;
				case GETS: message = "Number of gets: "+item.getGets();
				break;
				case PUTS: message = "Number of puts: "+item.getPuts();
				break;
				case REMOVES: message = "Number of removes: "+item.getRemoves();
				break;
										
				default: message = "Entries: "+item.getEntries();						
										
				}

				return message;
			}
		});
	    series.setToolTipConfig(config);

	}

	/**
	 * 
	 */
	private void createLegend() {
		final Legend<Data> legend = new Legend<Data>();
		legend.setPosition(Position.RIGHT);
//		legend.setItemHighlighting(true);
//		legend.setItemHiding(true);
		chart.setLegend(legend);
	}
}