/**
    JEM, the BEE - Job Entry Manager, the Batch Execution Environment
    Copyright (C) 2012, 2013   Andrea "Stock" Stocchero
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
package org.pepstock.jem.gwt.client.panels;

import java.util.Collection;

import org.pepstock.jem.Job;
import org.pepstock.jem.OutputTree;
import org.pepstock.jem.gwt.client.commons.AlertMessageBox;
import org.pepstock.jem.gwt.client.commons.Loading;
import org.pepstock.jem.gwt.client.commons.SearchListener;
import org.pepstock.jem.gwt.client.commons.ServiceAsyncCallback;
import org.pepstock.jem.gwt.client.commons.UpdateListener;
import org.pepstock.jem.gwt.client.events.EventBus;
import org.pepstock.jem.gwt.client.events.FilterEvent;
import org.pepstock.jem.gwt.client.events.FilterEventHandler;
import org.pepstock.jem.gwt.client.panels.components.BasePanel;
import org.pepstock.jem.gwt.client.panels.components.CommandPanel;
import org.pepstock.jem.gwt.client.panels.components.TableContainer;
import org.pepstock.jem.gwt.client.panels.jobs.commons.JobInspector;
import org.pepstock.jem.gwt.client.panels.jobs.commons.JobsSearcher;
import org.pepstock.jem.gwt.client.panels.jobs.running.RunningActions;
import org.pepstock.jem.gwt.client.panels.jobs.running.RunningTable;
import org.pepstock.jem.gwt.client.security.PreferencesKeys;
import org.pepstock.jem.gwt.client.services.Services;
import org.pepstock.jem.log.Message;
import org.pepstock.jem.node.Queues;

import com.google.gwt.core.client.Scheduler;
import com.google.gwt.core.client.Scheduler.ScheduledCommand;
import com.google.gwt.event.logical.shared.CloseEvent;
import com.google.gwt.event.logical.shared.CloseHandler;
import com.google.gwt.user.client.ui.PopupPanel;

/**
 * Main panel of job RUNNING queue manager. Shows the list of jobs in execution with the possibilities to act on them. 
 * Furthermore allows to inspect the job to see JCL and general information.
 * 
 * @author Andrea "Stock" Stocchero
 * 
 */
public class Running extends BasePanel<Job> implements SearchListener, UpdateListener<Job> {

	private final Loading loading = new Loading();

	/**
	 *  Constructs all UI 
	 */
	public Running() {
		super(new TableContainer<Job>(new RunningTable(true)),
				new CommandPanel<Job>(new JobsSearcher(PreferencesKeys.JOB_SEARCH_RUNNING), new RunningActions()));
		
		// sets listeners
		getCommandPanel().getSearcher().setSearchListener(this);
		getCommandPanel().getActions().setUnderlyingTable(getTableContainer().getUnderlyingTable());
		getTableContainer().getUnderlyingTable().setInspectListener(this);

		// subscribe the basic filter event handler to eventbus
		EventBus.INSTANCE.addHandler(FilterEvent.TYPE, (FilterEventHandler)getCommandPanel().getSearcher());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see test.client.main.JobsSearchListener#search(java.lang.String)
	 */
	@Override
	public void search(final String jobsFilter) {
		getCommandPanel().getSearcher().setEnabled(false);
		loading.startProcessing();

	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				Services.QUEUES_MANAGER.getRunningQueue(jobsFilter, new ServiceAsyncCallback<Collection<Job>>() {

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Search error!", 
								caught.getMessage());
						alert.open();
						getCommandPanel().getSearcher().setFirstSearch(true);
					}

					@Override
					public void onJemSuccess(Collection<Job> result) {
						// sets data to table to show it
						getTableContainer().getUnderlyingTable().setRowData(result);
						getCommandPanel().getSearcher().setFirstSearch(false);
					}
					
					@Override
                    public void onJemExecuted() {
						loading.stopProcessing();
						getCommandPanel().getSearcher().setEnabled(true);
                    }

				});
			
			}
	    });

		
	}

	/**
	 * @see org.pepstock.jem.gwt.client.panels.jobs.commons.JobInspectListener#inspect(org.pepstock.jem.Job)
	 */
	@Override
	public void inspect(final Job job) {
		getCommandPanel().getSearcher().setEnabled(false);
		loading.startProcessing();
	    Scheduler scheduler = Scheduler.get();
	    scheduler.scheduleDeferred(new ScheduledCommand() {
			
			@Override
			public void execute() {
				// asks for output tree with JCL content
				Services.QUEUES_MANAGER.getOutputTree(job, Queues.RUNNING_QUEUE, new ServiceAsyncCallback<OutputTree>() {

					@Override
					public void onJemFailure(Throwable caught) {
						AlertMessageBox alert = new AlertMessageBox(Message.ERROR, "Get OUTPUT error!", 
								caught.getMessage());
						alert.open();
					}

					@Override
					public void onJemSuccess(OutputTree result) {
						//sets JCL content
						job.getJcl().setContent(result.getJclContent());
						
						// creates the inspector and shows it
						JobInspector inspector = new JobInspector(job, result);
						inspector.setModal(true);
						inspector.setTitle(job.getName());
						inspector.center();
						
						// adds for closing
						inspector.addCloseHandler(new CloseHandler<PopupPanel>() {
							
							@Override
							public void onClose(CloseEvent<PopupPanel> arg0) {
								// performs a refresh on the table
								getCommandPanel().getSearcher().refresh();
							}
						});
					}
					
					@Override
                    public void onJemExecuted() {
						loading.stopProcessing();
						getCommandPanel().getSearcher().setEnabled(true);
                    }

				});
			
			}
	    });
		
	}

	/* (non-Javadoc)
	 * @see org.pepstock.jem.gwt.client.panels.jobs.commons.JobInspectListener#update(org.pepstock.jem.Job)
	 */
    @Override
    public void update(Job object) {
	    // do nothing
    }

}